# -*- coding: utf-8 -*-
"""
Part of
7log: cmd line logbook
written by dg7bbp

(c) 2022, dg7bbp, Jens Rosebrock

Reading repeater list
Find next repeater
Uses repeater list from www.darc.de/a23
https://www.a23-wertheim.de/downloads/d-star/repeater-listen


"""

import os
import glob
from collections import defaultdict
from locator import Point, distance_haversine, toLoc


class Repeater(object):
    def __init__(self, files):
        """
        files: list of filenames to read
        """
        # frequenz zu liste von repeatern
        self.frequency = defaultdict(list)
        self.repeater = []  # list of repeater dicts
        for f in files:
            self.repeater.extend(list(self.read_csv(f)))
        for r in self.repeater:
            f_str = r["Frequency"]
            f = float(f_str)
            self.frequency[f].append(r)

    def read_csv(self, repeater_file):
        """
        reads repeater list from repeater_file

        Group No;Group Name;Name;Sub Name;Repeater Call Sign;Gateway Call Sign;Frequency;Dup;Offset;Mode;TONE;Repeater Tone;RPT1USE;Position;Latitude;Longitude;UTC Offset

        """
        with open(repeater_file, "r") as f:
            fields = None
            has_mode = False
            for l in f.readlines():
                line = l.strip()
                spLine = line.split(";")
                if fields is None:
                    fields = spLine
                    has_mode = "Mode" in fields
                else:
                    r = dict()
                    for i, k in enumerate(fields):
                        r[k] = spLine[i]
                    if not has_mode:
                        r["Mode"] = "DV"
                    yield r

    def by_distance(self, location, by_freq=None):
        """
        :param location: Point
        :param by_freq: float in Mhz
        :returns dict(distance)->list of repeaters
        """
        dist_dict = defaultdict(list)
        for r in self.repeater:
            if by_freq is None or float(r["Frequency"]) == by_freq:
                lat = float(r["Latitude"])
                lon = float(r["Longitude"])
                rp = Point(lon, lat)
                dist = distance_haversine(location, rp)
                dist_dict[dist].append(r)
        return dist_dict


def _test():
    searchdir = os.path.join(os.path.dirname(__file__), "repeater", "*.csv")
    fnames = list(glob.glob(searchdir))
    r = Repeater(fnames)
    distances = r.by_distance(toLoc("JO43KB"), None)
    d_sort = sorted(distances.keys())
    for i in range(min(len(d_sort), 50)):
        d = d_sort[i]
        repeaters = distances[d]
        calls = ", ".join([r["Repeater Call Sign"] for r in repeaters])
        calls = ", ".join(
            [
                r["Frequency"]
                + " "
                + r["Mode"]
                + ": "
                + r["Name"]
                + "/"
                + r["Sub Name"]
                for r in repeaters
            ]
        )
        print("Dist: {} km: {}".format(int(int(d) / 1e3), calls))


if __name__ == "__main__":
    _test()
