#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
/log: cmd line logbook
written by dg7bbp

(c) 2019, dg7bbp, Jens Rosebrock
"""
import re
import requests
import json
import os
from xml.etree import ElementTree as ET
from logconfig import get_key_file


class QRZException(Exception):
    pass


class QrzAccess(object):

    base_url = "://xmldata.qrz.com/xml/"

    def __init__(self, user=None, pw=None):
        self.session_key = None
        self.user = user
        self.pw = pw
        self._read_key()

    def login(self, user, passwd):
        self.session_key = None
        p = {"username": user, "password": passwd, "agent": "7log"}
        print("Relogin in qrz.com, session expired")
        r = requests.post("https" + self.base_url, data=p)
        if r.status_code == 200:
            root = ET.fromstring(r.text)
            for c in root:
                if c.tag.lower().endswith("session"):
                    for sc in c:
                        if sc.tag.lower().endswith("key"):
                            self.session_key = sc.text
                            self._write_key()
                            return True
                        elif c.tag.lower().endswith("error") and c.text:
                            return c.text
                elif c.tag.lower().endswith("error"):
                    return c.text
        else:
            print("Connection error from qrz.com: {} {}".format(r.status_code, r.text))
        return False

    def call_info(self, call_with_extension):
        """ """
        call_parts = call_with_extension.split("/")
        if len(call_parts[0]) < 3:
            # like PA/DG7BBP/P or oz/dg7bbp
            call = call_parts[1]
        else:
            # only strip /m or /p
            call = call_parts[0]
        is_session_failure = False
        info = None
        if not self.session_key:
            if not self.login(self.user, self.pw):
                print("qrz.com: login failure")
        try:
            info = self._call_info(call)
        except QRZException as e:
            is_session_failure = str(e).lower().find("session") >= 0
            if not is_session_failure:
                raise
        # don't know the error,
        # if known only on session key expired
        # and not on unknown callsign
        if not info and is_session_failure:
            if self.login(self.user, self.pw):
                info = self._call_info(call)
            else:
                print("qrz.com: login failure")
        return info

    def _call_info(self, call):
        """
        :returns dict with info.
        without subscription at least fname, name and country
        """
        p = {"s": self.session_key, "callsign": call}
        vals = {}
        if self.session_key:
            r = requests.get("http" + self.base_url, params=p)
            if r.status_code == 200:
                root = ET.fromstring(r.text)
                for c in root:
                    if c.tag.lower().endswith("callsign"):
                        for sc in c:
                            if sc.text:
                                k = re.search(r"\{.*\}(.*)$", sc.tag)
                                if k:
                                    key = k.group(1)
                                else:
                                    key = sc.tag
                                vals[key] = sc.text
                    elif c.tag.lower().endswith("session"):
                        for sc in c:
                            if sc.tag.lower().endswith("error"):
                                raise QRZException(sc.text)
            else:
                print(
                    "Connection error from qrz.com: {} {}".format(r.status_code, r.text)
                )
        return vals

    def _read_key(self):
        fname = get_key_file()
        if os.path.isfile(fname):
            with open(fname, "r") as f:
                d = json.load(f)
                if d:
                    s_key = d.get("qrz_session_key")
                    if s_key:
                        self.session_key = s_key

    def _write_key(self):
        fname = get_key_file()
        if self.session_key:
            k = {"qrz_session_key": self.session_key}
            with open(fname, "w") as f:
                json.dump(k, f)
        elif os.path.isfile(fname):
            os.unlink(fname)


if __name__ == "__main__":
    q = QrzAccess("mycall", "mypw")
    print(q.call_info("dd0kp"))
    print(q.call_info("tf1a"))
