#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019, dg7bbp, Jens Rosebrock

parameter guessing machine
"""

import re
import time

_regexps = [
    ("locator", "^[A-R]{2}[0-9]{2}[A-Z]{2,4}$"),
    ("freq", "^[0-9]+\.*[0-9]*[A-Z]HZ$"),
    ("report_rcvd", "^[0-5][0-9]{1,2}/{0,1}[A-Z0-9]*$"),
    ("time", "^[0-9]{2}:[0-9]{2}$"),
    ("date", "^[0-9]{4}-[0-9]{2}-[0-9]{2}$"),
    (
        "mode",
        "^SSB|USB|LSB|CW|CODEC2|FM|DRM|C4|RTTY|PSK|FT8|FT4|JT65.*([A-Z][0-9][A-Z])$",
    ),
    ("qslin", "^QSLIN$"),
    ("qslout", "^QSLOUT$"),
    ("qslreq", "^QSLREQ$"),
    ("call", "^([A-Z]{1,2}|[0-9][A-Z])[0-9]{1,}[A-Z].*[/]*[A-Z]*"),
]

_named_args = {
    "n:": "name",
    "s:": "report_send",
    "r:": "repeater",
    "c:": "call",
    "/": "contest_ex_rx",
    "b:": "band",
}


def guess_args(args):
    """
    :retuns dict of matched args, list of unmatched args
    """
    unmatched = []
    matched = {}
    unnamed = []

    for arg in args:
        for n, k in _named_args.items():
            if arg.startswith(n):
                matched[k] = arg[len(n) :]
                break
        else:
            unnamed.append(arg)
    for arg in unnamed:
        arg_ok = False
        for k, pattern in _regexps:
            # print("Test : %s %s", k, arg)
            if re.match(pattern, arg.upper()):
                # print("Matches : %s %s", k, arg)
                matched[k] = arg
                arg_ok = True
                break
        if not arg_ok:
            unmatched.append(arg)
    return matched, unmatched


def time_defaults():
    """
    returns dict with time default from current time
    """
    gm_time = time.gmtime()
    date_str = "%04d-%02d-%02d" % gm_time[:3]
    time_str = "%02d:%02d" % gm_time[3:5]
    time_defaults = {"date": date_str, "time": time_str}
    return time_defaults


def frequency_to_band(f):
    _bands = [
        ("2190m", 0.1357, 0.1378),
        ("630m", 0.472, 0.479),
        ("560m", 0.501, 0.504),
        ("160m", 1.8, 2.0),
        ("80m", 3.5, 4.0),
        ("60m", 5.06, 5.45),
        ("40m", 7.0, 7.3),
        ("30m", 10.1, 10.15),
        ("20m", 14.0, 14.35),
        ("17m", 18.068, 18.168),
        ("15m", 21.0, 21.450),
        ("12m", 24.890, 24.99),
        ("10m", 28.0, 29.7),
        ("6m", 50.0, 54.0),
        ("4m", 70.0, 71.0),
        ("2m", 144.0, 148.0),
        ("70cm", 420.0, 450.0),
        ("23cm", 1240.0, 1300.0),
        ("13cm", 2300.0, 2450.0),
        ("6cm", 5650.0, 5925.0),
        ("3cm", 10000.0, 10500.0),
        ("1.25cm", 24000.0, 24250),
        ("6mm", 47000.0, 47200.0),
    ]
    band = "UNKOWN"
    for b, l, u in _bands:
        if f >= l * 1e6 and f <= u * 1e6:
            band = b
            break
    return band


class logAccess(object):

    db_names = {
        "call": "dst_call",
        "locator": "dst_locator",
        "date": "begin_date",
        "time": "begin_time",
        "finished": "end_time",
        "report_send": "rst_send",
        "report_rcvd": "rst_received",
        "info": "comment",
    }

    def __init__(self, db, const_values):
        self.db = db
        self.const_values = const_values
        # const values is dict with default values like current op, session,...

    def process_input(self, named_args, unnamed_args):
        """
        process named args (dialog items) and tries to
        match unnamed args
        update db record or inserts new record if call changes
        and previous qso is complete
        returns db_record
        """
        pass


def _test():
    assert guess_args(["n:Jens", "dg7bbp"])[0]["name"] == "Jens"
    assert guess_args(["n:Jens", "dg7bbp"])[0]["call"] == "dg7bbp"
    assert guess_args(["2.4013Ghz"])[0]["freq"] == "2.4013Ghz"
    assert guess_args(["10:22"])[0]["time"] == "10:22"
    assert guess_args(["dg7bbp"])[0]["call"] == "dg7bbp"
    assert guess_args(["K9AN"])[0]["call"] == "K9AN"
    assert guess_args(["9A7L"])[0]["call"] == "9A7L"
    assert guess_args(["jo43la"])[0]["locator"] == "jo43la"
    assert guess_args(["SSB"])[0]["mode"] == "SSB"

    assert guess_args(["599a"])[0]["report_rcvd"] == "599a"

    assert guess_args(["2020-10-03"])[0]["date"] == "2020-10-03"
    assert guess_args(["Jens"])[1][0] == "Jens"

    t_def = time_defaults()
    assert len(t_def["date"]) == 10
    assert len(t_def["time"]) == 5

    assert frequency_to_band(3.750e6) == "80m"
    print("Ready")


if __name__ == "__main__":
    _test()
