#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019, dg7bbp, Jens Rosebrock

Handle configuration values
"""

import sys
import json
import os
import re
import glob

conf_file = "7log.conf"
rig_conf = "rigdesc.json"
cloudlog_conf = "cloudlog.json"
contest_conf = "contesttemplate.json"
session_file = "sessions.key"
default_db = "7log.sqlite"
linux_path = [".config", "7log"]
windows_path = ["dg7bbp", "7log"]
repeater_dir = "repeater"

_upper_fields = ["callsign", "operator"]


def get_conf_directory():
    p = ""
    if sys.platform.startswith("win"):
        p = os.path.join(os.environ.get("LOCALAPPDATA"), *windows_path)
    elif sys.platform.startswith("linux"):
        p = os.path.join(os.environ.get("HOME"), *linux_path)
    return p


def get_repeater_files():
    """
    returns list of all files  asabpath in repeater
    directory
    """
    rep_files = ()
    conf_dir = get_conf_directory()
    rep_dir = os.path.join(conf_dir, repeater_dir)
    if os.path.isdir(rep_dir):
        rep_files = glob.glob(os.path.join(rep_dir, "*.csv"))
    return rep_files


def get_conf_filename():
    p = get_conf_directory()
    fname = os.path.join(p, conf_file)
    return fname


def get_key_file():
    p = get_conf_directory()
    fname = os.path.join(p, session_file)
    return fname


def get_rig_config():
    p = get_conf_directory()
    fname = os.path.join(p, rig_conf)
    return fname


def get_cloudlog_config():
    p = get_conf_directory()
    fname = os.path.join(p, cloudlog_conf)
    return fname


def get_db_path():
    p = get_conf_directory()
    fname = os.path.join(p, default_db)
    return fname


def get_contest_config():
    p = get_conf_directory()
    fname = os.path.join(p, contest_conf)
    return fname


def read_config(defaults=False):
    """
    read config with default values from disk
    """
    db_path = get_db_path()
    data_def = {
        "session_id": "",
        "callsign": "",
        "operator": "",
        "locator": "",
        "database": db_path,
        "lastqsos": 1,
        "qrzlogin": "",
        "qrzpassword": "",
        "rigctlhost": "",
        "defaultrig": "",
        "dxclusterhost": "",
        "dxclusterport": "",
        "dxclustercall": "",
        "cloudlogurl": "",
        "cloudlogkey": "",
        "cloudlogstationid": "",
    }
    data = {}
    data.update(data_def)
    if not defaults:
        fname = get_conf_filename()
        if os.path.isfile(fname):
            with open(fname) as f:
                data = json.load(f)
                # update new entries
                for k, v in data_def.items():
                    if k not in data:
                        data[k] = v
    return data


def _write_config(data):
    """
    Write config values to disk
    """
    fname = get_conf_filename()
    conf_dir = get_conf_directory()
    if not os.path.isdir(conf_dir):
        os.makedirs(conf_dir, mode=0o755)
    with open(fname, "w") as f:
        json.dump(data, f)


def set_config_value(key, value):
    data = read_config()
    if key in _upper_fields:
        data[key] = value.upper()
    else:
        data[key] = value
    _write_config(data)


class RigDesc(object):
    """
    {"ic705home": {"rigdesc": "IC 705 controlled by Raspberry PI",
               "antenna": {"default": "Dipol 2 * 7m with BX 1200 tuner",
                          "2m": "X-5000",
                          "70cm": "X-5000"},
               "rigctldhost": "localhost:4532",
               "freqoffset": 0.0},
               "rotcltdhost": "localhost:4533"
    """

    def __init__(self, name, rigcontent):
        self.name = name
        self.rigcontent = rigcontent

    def get_antenna(self, band):
        adict = self.rigcontent.get("antenna")
        antenna = ""
        if adict:
            d = adict.get("default", "")
            antenna = adict.get(band, d)
        return antenna

    def get_rigdesc(self):
        return self.rigcontent.get("rigdesc", self.name)

    def rigctrl(self):
        return self.rigcontent.get("rigctldhost", "localhost:4532")

    def rotctrl(self):
        return self.rigcontent.get("rotctldhost", "")

    def isValid(self):
        return bool(self.name) and bool(self.rigcontent)

    def frequency_offset(self):
        return self.rigcontent.get("freqoffset", 0.0)


def read_rig(rigname):
    """
    reads rigdescription from ~/.config/7log/rigdesc.json
    returns: RigDesc
    """
    rigconf_name = get_rig_config()
    if os.path.isfile(rigconf_name):
        with open(rigconf_name) as f:
            config = json.load(f)
            current_config = config.get(rigname, {})
    else:
        current_config = {}
    return RigDesc(rigname, current_config)


def rig_names():
    rignames = []
    rigconf_name = get_rig_config()
    if os.path.isfile(rigconf_name):
        with open(rigconf_name) as f:
            config = json.load(f)
            rignames = list(config)
    return rignames


class InvalidContestDesc(Exception):
    pass


class ContestDesc(object):
    """
    {"<name>": {"number_per_band": Bool,
            "number_scheme": String ($(NUM) is replaced by current number. else const,
            "exchange_data": Like number scheme but only as information
            "required_attrs": ["locator, name", "...],
            "unique_station_attrs": ["call", "band", "mode"] list for double qsos
    """

    def __init__(self, name, contest_desc):
        self.name = name
        self.number_per_band = contest_desc.get("number_per_band")
        self.number_scheme = contest_desc.get("number_scheme")
        self.exchange_data = contest_desc.get("exchange_data", self.number_scheme)
        self.required_attrs = contest_desc.get("required_attrs")
        self.unique_station_attrs = contest_desc.get("unique_station_attrs")
        if not self.name:
            raise InvalidContestDesc("Contest_desc needs name")
        if not isinstance(self.number_per_band, bool):
            raise InvalidContestDesc(
                "Contest: {}:{} is invalid".format(self.name, "number_per_band")
            )
        if not self.number_scheme:
            raise InvalidContestDesc(
                "Contest: {}:{} is invalid".format(self.name, "number_scheme")
            )
        if not self.required_attrs:
            raise InvalidContestDesc(
                "Contest: {}:{} is invalid".format(self.name, "required_attrs")
            )
        if not self.unique_station_attrs:
            raise InvalidContestDesc(
                "Contest: {}:{} is invalid".format(self.name, "unique_station_attrs")
            )

    def scheme_variables(self):
        """
        return variable names like $(NUM) all uppercase
        """
        svars = set()
        if self.number_scheme:
            svars.update(
                set(re.findall(r"\$\([a-z]+\)", self.number_scheme.upper(), re.I))
            )
        if self.exchange_data:
            svars2 = re.findall(r"\$\([a-z]+\)", self.exchange_data.upper(), re.I)
            svars.update(svars2)
        return list(svars)


def read_contest_templates():
    """
    Read contests from disc and returns
    dict name-> ContestDesc
    """
    contests = {}
    # read standard contests from program dir
    def_name = os.path.join(os.path.dirname(__file__), "contests.json")
    contests.update(read_contests_from_file(def_name))
    c_name = get_contest_config()
    contests.update(read_contests_from_file(c_name))
    return contests


def read_contests_from_file(c_name):
    contests = {}

    if c_name and os.path.isfile(c_name):
        with open(c_name) as f:
            configs = json.load(f)
        for name, vals in configs.items():
            contests[name] = ContestDesc(name, vals)
    return contests


def country_data_files():
    conf_dir = get_conf_directory()
    country_files = ["AreaOK1RR.tbl", "Country.tab"]
    return [os.path.join(conf_dir, f) for f in country_files]


def get_cloud_log(name):
    if name.lower() == "default":
        config = read_config()
        cl_config = {
            "url": config.get("cloudlogurl"),
            "key": config.get("cloudlogkey"),
            "stationid": config.get("cloudlogstationid"),
        }
        if not cl_config["url"] or not cl_config["key"] or not cl_config["stationid"]:
            print("Invalid defaut cloudlog", file=sys.stderr)
            cl_config = None
    else:
        conf_path = get_cloudlog_config()
        data = {}
        if os.path.isfile(conf_path):
            with open(conf_path) as f:
                data = json.load(f)
        cl_config = data.get(name)
        if cl_config is not None:
            if not cl_config.get("url"):
                print("Missing url in cloudlog configuration", file=sys.stderr)
                cl_config = None
        if cl_config is not None:
            if not cl_config.get("key"):
                cl_config = None
                print("Missing key in cloudlog configuration", file=sys.stderr)
        if cl_config is not None:
            if not cl_config.get("stationid"):
                cl_config = None
                print("Missing stationid in cloudlog configuration", file=sys.stderr)
    return cl_config
