# -*- coding: utf-8 -*-
"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019, dg7bbp, Jens Rosebrock

"""
from typing import Tuple
from math import radians, degrees, cos, sin, atan2, sqrt, floor

# http://www.samuelbosch.com/2014/05/working-in-lat-long-great-circle.html


class Point(object):
    def __init__(self, x, y):
        self.x = x
        self.y = y


def distance_haversine(A, B, radius=6371000):
    """note that the default distance is in meters"""
    dLat = radians(B.y - A.y)
    dLon = radians(B.x - A.x)
    lat1 = radians(A.y)
    lat2 = radians(B.y)
    a = sin(dLat / 2) * sin(dLat / 2) + sin(dLon / 2) * sin(dLon / 2) * cos(lat1) * cos(
        lat2
    )
    c = 2 * atan2(sqrt(a), sqrt(1 - a))
    return c * radius


def bearing(A, B):
    dLon = radians(B.x - A.x)
    lat1 = radians(A.y)
    lat2 = radians(B.y)
    y = sin(dLon) * cos(lat2)
    x = cos(lat1) * sin(lat2) - sin(lat1) * cos(lat2) * cos(dLon)
    return atan2(y, x)


def bearing_degrees(A, B):
    return (degrees(bearing(A, B)) + 360.0) % 360.0


def toLoc(maiden: str) -> Tuple[float, float]:
    """
    convert Maidenhead grid to latitude, longitude
    Parameters
    ----------
    maiden : str
        Maidenhead grid locator of length 2 to 8
    Returns
    -------
    latLon : tuple of float
        Geographic latitude, longitude
    """
    if not isinstance(maiden, str):
        raise TypeError("Maidenhead locator must be a string")

    maiden = maiden.strip().upper()

    N = len(maiden)
    if not 8 >= N >= 2 and N % 2 == 0:
        raise ValueError(
            "Maidenhead locator requires 2-8 characters, even number of characters"
        )

    Oa = ord("A")
    lon = -180.0
    lat = -90.0
    # %% first pair
    lon += (ord(maiden[0]) - Oa) * 20
    lat += (ord(maiden[1]) - Oa) * 10
    # %% second pair
    if N >= 4:
        lon += int(maiden[2]) * 2
        lat += int(maiden[3]) * 1
    # %%
    if N >= 6:
        lon += (ord(maiden[4]) - Oa) * 5.0 / 60.0
        lat += (ord(maiden[5]) - Oa) * 2.5 / 60.0
        if N < 8:  # use center of field
            lon += 2.5 / 60.0
            lat += 1.25 / 60.0
    # %%
    if N >= 8:
        lon += int(maiden[6]) * 5.0 / 600.0
        lat += int(maiden[7]) * 2.5 / 600.0

    return Point(lon, lat)


def nordcontest(p1, p2):
    """
    Points for nordcontest
    """
    diffx = int(abs(floor(p1.x) - floor(p2.x)) / 2.0)
    diffy = int(abs(floor(p1.y) - floor(p2.y)))
    points = max(diffx, diffy) + 1
    return points


def test_():
    A = toLoc("JO43KA")
    B = toLoc("JO43LA")
    print(bearing_degrees(A, B))
    print(distance_haversine(A, B) / 1e3)
    C = toLoc("JO44QQ")
    D = toLoc("JO45AA")
    E = toLoc("JO32LL")
    F = toLoc("JO20QQ")
    assert nordcontest(A, B) == 1
    assert nordcontest(A, C) == 2
    assert nordcontest(A, E) == 2
    assert nordcontest(D, E) == 4
    assert nordcontest(A, F) == 4


if __name__ == "__main__":
    test_()
