#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019, dg7bbp, Jens Rosebrock

Hamlibconnection
"""

import socket
import logging
import time

EOT = "\x04"
LF = "\x0A"
_verbose_level = -1


def log(level, msg):
    logging.debug(msg)


class HamLibConnection(object):
    """
    Simple class for requesting frequency from rigctld or rotctld
    """

    def __init__(self, host_port):
        self._host_port = host_port
        self._socket = None

    def __enter__(self):
        """
        Allow with statement
        """
        self.connect()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """ """
        self.disconnect()

    def connect(self, retry=1):
        """
        Connect to rightld
        :raises socket.error
        """
        addr_data = self.url_to_host(self._host_port, 4532)
        if len(addr_data) == 2:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        else:
            s = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        last_err = None
        for i in range(retry):
            last_err = None
            try:
                s.connect(addr_data)
                break
            except (socket.error,) as e:
                last_err = e
                time.sleep(1)
        if last_err is None:
            self._socket = s
        else:
            raise last_err

    def disconnect(self):
        """
        Disconnect from rigtld
        """
        if self._socket is not None:
            self._socket.close()

    def url_to_host(self, host_port, default_port):
        """
        get hostname, ip address from host_port string like:
        <hostname>[:<port>]
        <ipv4_addr>[:<port>]
        \[<ipv6_addr>\][:<port>]
        Port is optional
        """
        splitted = host_port.split(":")
        port_str = None
        is_ipv6 = False
        if len(splitted) == 1:
            # ip v4/hostname no port
            host = host_port
        elif len(splitted) == 2:
            # ip v4/hostname given port
            host = splitted[0]
            port_str = splitted[1]
        else:
            log(3, "url_to_host: IPV6 %s" % (splitted,))
            is_ipv6 = True
            # some kind of ip6
            if splitted[-1].endswith("]"):
                # no port
                log(3, "url_to_host: NO PORT")
                host = host_port
                if host.startswith("[") and host.endswith("]"):
                    host = host[1:-1]
            else:
                port = splitted[-1]
                host = ":".join(splitted[:-1])
                if host.startswith("[") and host.endswith("]"):
                    host = host[1:-1]
        if port_str is None or not port_str.isdigit():
            port = default_port
        else:
            port = int(port_str)
        log(3, "url_to_host: HOST %s PORT: %s %s" % (host, port, is_ipv6))
        if is_ipv6:
            return host, port, 0, 0
        else:
            return host, port

    def read_frequency(self):
        """
        Simple read function for reading the current frequency in Hz
        from rigctld. No special error handling is supplied.

        :return: None if error else  float frequency in Hz
        """
        result = None
        if self._socket:
            self._socket.sendall(self._encode("f" + chr(13)))
            data = self._decode(self._socket.recv(1000))
            log(3, "hamlib_read_frequency: %s" % data)
            if data:
                if not data.startswith("RPRT"):
                    result = float(data.strip())
        return result

    def set_frequency(self, frequency):
        """
        sets the frequency of connected trx
        :return Boolean
        """
        result = False
        if self._socket:
            log(3, "hamlib_set_frequency: %s" % frequency)
            self._socket.sendall(self._encode("F %d" % int(frequency) + chr(13)))
            data = self._decode(self._socket.recv(1000))
            if data:
                result = data.startswith("RPRT 0")
        return result

    def set_split_frequency(self, frequency):
        """
        sets the frequency of connected trx
        :return Boolean
        """
        result = False
        if self._socket:
            log(3, "hamlib_set_split_frequency: %s" % frequency)
            self._socket.sendall(self._encode("I %d" % int(frequency) + chr(13)))
            data = self._decode(self._socket.recv(1000))
            if data:
                result = data.startswith("RPRT 0")
        return result

    def _encode(self, ustr):
        return ustr.encode("utf-8")

    def _decode(self, bytestr):
        decoded = None
        if bytestr is not None:
            decoded = bytestr.decode("utf-8")
        return decoded

    def get_mode(self):
        """
        :return (string, string)  mode, bandwidth of transceiver
        """
        result = None
        if self._socket:
            self._socket.sendall(self._encode("m" + chr(13)))
            data = self._decode(self._socket.recv(1000))
            log(3, "hamlib_read_mode: %s" % data)
            if data:
                if not data.startswith("RPRT"):
                    result = data.strip().split()
        return result

    def rot_set_pos(self, az, el):
        """
        set rotor position
        """
        result = False
        if self._socket:
            log(3, "hamlib_rot_set_pos %s %s" % (az, el))
            self._socket.sendall(
                self._encode("P %.2f %f.2" % (float(az), float(el)) + chr(13))
            )
            data = self._decode(self._socket.recv(1000))
            if data:
                result = data.startswith("RPRT 0")
        return result

    def rot_get_pos(self):
        """
        requests current rotator position
        :return (az as float, el as float)
        """
        az = None
        el = None
        if self._socket:
            self._socket.sendall(self._encode("p" + chr(13)))
            data = self._decode(self._socket.recv(1000))
            log(3, "hamlib_read_pos: %s" % data)
            if data:
                if not data.startswith("RPRT"):
                    splitted = data.strip().split()
                    if len(splitted) > 0:
                        az = float(splitted[0])
                    if len(splitted) > 1:
                        el = float(splitted[1])
        return az, el
