# -*- coding: utf-8 -*-

"""
/log: cmd line logbook
written by dg7bbp

Initially developed for gqrx_set_trx.py

(c) 2019-2021, dg7bbp, Jens Rosebrock
""" ""
from locator import toLoc, distance_haversine, nordcontest
import re
import os
from logconfig import get_conf_directory


class EDITYPE(object):
    EDI_NORD = 1
    EDI_VUHF = 0


_vars = {
    "CALL": None,
    "LOCATOR": None,
    "MHZ": None,
    "BEGINDATE": None,
    "ENDDATE": None,
    "RIG": None,
    "ANTENNA": None,
    "QSOCNT": None,
    "QSOPOINTS": None,
    "LOCATORCNT": None,
    "ODXCALL": None,
    "ODXLOC": None,
    "ODXPOINTS": None,
}


class EdiQSO(object):
    @classmethod
    def contestdate(cls, dbdate, yyyy=False):
        """
        dbdate is iso
        """
        if yyyy:
            sy = 0
        else:
            sy = 2
        cdate = "{}{}{}".format(dbdate[sy:4], dbdate[5:7], dbdate[8:10])
        return cdate

    @classmethod
    def contestmode(cls, dbmode):
        cmode = 0
        if dbmode in ["SSB", "USB", "LSB"]:
            cmode = 1
        elif dbmode == "CW":
            cmode = 2
        return cmode

    def __init__(self, dbrow):
        self.db = dbrow
        self.date = self.contestdate(dbrow["begin_date"])
        self.time = dbrow["begin_time"]
        self.call = dbrow["dst_call"]
        self.mode = self.contestmode(dbrow["mode"])
        self.rcv_exchange = ""
        self.dstlocator = dbrow["dst_locator"]
        A = toLoc(dbrow["src_locator"])
        B = toLoc(self.dstlocator)
        self.points = int(distance_haversine(A, B) / 1e3)
        s = dbrow["rst_send"]
        r = dbrow["rst_received"]
        if self.mode == 2:  #  CW
            self.s_number = int(s[3:])
            self.s_rst = s[0:3]
            self.r_number = int(r[3:])
            self.r_rst = r[0:3]
        else:  # SSB
            self.s_number = int(s[2:])
            self.s_rst = s[0:2]
            if r[2:] and r[2:].isdigit():
                self.r_number = int(r[2:])
                self.r_rst = r[0:2]
            else:
                self.r_number = 0
                self.r_rst = ""
                self.points = 0
                self.call = "ERROR"

    def qsodata(self):
        """
        :returns list in follwing order
        date, time, call, mode, send rst, send number, rcv rst,
        rcv number , rcv_exchange, rcv loc, qsopoints, "", "M:NEWLoc","N:New DXCC", "(Dup)"
        """
        r = [
            self.date,
            self.time,
            self.call,
            str(self.mode),
            self.s_rst,
            "{:03d}".format(self.s_number),
            self.r_rst,
            "{:03d}".format(self.r_number),
            self.rcv_exchange,
            self.dstlocator,
            str(self.points),
            "",
            "",
            "",
            "",
        ]
        return r


class EdiQSONord(EdiQSO):
    def __init__(self, dbrow):
        EdiQSO.__init__(self, dbrow)
        A = toLoc(dbrow["src_locator"])
        B = toLoc(self.dstlocator)
        self.points = nordcontest(A, B)
        self.rcv_exchange = dbrow["comment"]


class EdiExporter(object):
    def __init__(self, db, contestname, edi_template, file_prefix):
        """
        param edi_template: filename
        """
        self.db = db
        self.contestname = contestname
        self.edi_template = edi_template
        self.file_prefix = file_prefix

    def bands(self):
        """
        return list of bands used in contest
        """
        sql = "select distinct band from qsos where contest_name=?"
        bands = [r["band"] for r in self.db.sqlselect(sql, (self.contestname,))]
        return bands

    def bandtofreq(self, band):
        _bands = {
            "6m": "50 Mhz",
            "4m": "70 Mhz",
            "2m": "144 Mhz",
            "70cm": "432 Mhz",
            "23cm": "1,3 GHz",
            "13cm": "2,3 Ghz",
            "9cm": "3,4 Ghz",
            "6cm": "5,7 Ghz",
            "3cm": "10 Ghz",
        }
        return _bands.get(band, "--unknown--")

    def get_qso_data(self, band):
        """
        assumes same locator for all qsos
        return list of qso
        """
        sql = "select _rowid_, * from qsos where contest_name=? and band=? order by _rowid_"
        return [EdiQSO(r) for r in self.db.sqlselect(sql, (self.contestname, band))]

    def begin_date(self, band):
        sql = "select min(begin_date) as bd from qsos where contest_name=? and band=?"
        rows = list(self.db.sqlselect(sql, (self.contestname, band)))
        if rows:
            begin_date = EdiQSO.contestdate(rows[0]["bd"], yyyy=True)
        else:
            begin_date = None
        return begin_date

    def end_date(self, band):
        sql = "select max(begin_date) as ed from qsos where contest_name=? and band=?"
        rows = list(self.db.sqlselect(sql, (self.contestname, band)))
        if rows:
            end_date = EdiQSO.contestdate(rows[0]["ed"], yyyy=True)
        else:
            end_date = None
        return end_date

    def export_band(self, band):
        """
        loads data
        fills variables
        write to output
        """
        qso_recs = self.get_qso_data(band)
        bvars = {}
        if qso_recs:
            bvars["$(CALL)"] = qso_recs[0].db["src_call"]
            bvars["$(LOCATOR)"] = qso_recs[0].db["src_locator"]
            bvars["$(RIG)"] = qso_recs[0].db["rig"]
            bvars["$(ANTENNA)"] = qso_recs[0].db["antenna"]
            bvars["$(MHZ)"] = self.bandtofreq(band)
            bvars["$(BEGINDATE)"] = self.begin_date(band)
            bvars["$(ENDDATE)"] = self.end_date(band)
            bvars["$(QSOCNT)"] = str(len(qso_recs))
            qsos = self.get_qso_data(band)
            odx = 0
            odxEdi = None
            locators = set()
            totalpoints = 0
            for q in qsos:
                totalpoints += q.points
                if q.points > odx:
                    odxEdi = q
                    odx = q.points
                    locators.add(q.dstlocator[0:4])
            if odxEdi is not None:
                bvars["$(ODXCALL)"] = odxEdi.call
                bvars["$(ODXLOC)"] = odxEdi.dstlocator
                bvars["$(ODXPOINTS)"] = str(odxEdi.points)
                bvars["$(QSOPOINTS)"] = str(totalpoints)
                bvars["$(LOCATORCNT)"] = str(len(locators))
            export_name = "{}_{}.edi".format(self.file_prefix, band)
            with open(self.edi_template, "r") as f:
                with open(export_name, "w") as fout:
                    for l in f.readlines():
                        l_strip = l.strip()
                        l_rep = self.replace_vars(l_strip, bvars)
                        if l_rep:
                            fout.write("{}\n".format(l_rep))
                    for q in qsos:
                        fout.write("{}\n".format(";".join(q.qsodata())))

    def replace_vars(self, line, bvars):
        """ """
        svars = re.findall(r"\$\([a-z]+\)", line, re.I)
        for var in svars:
            varval = bvars.get(var)
            if varval is None:
                print("Invalid var in template '{}".format(var))
            else:
                line = line.replace(var, varval)
        return line

    def export(self):
        """
        run the export to file
        """
        for band in self.bands():
            self.export_band(band)


class EdiExporterNordContest(EdiExporter):
    def get_qso_data(self, band):
        """
        assumes same locator for all qsos
        return list of qso
        """
        sql = "select _rowid_, * from qsos where contest_name=? and band=? order by _rowid_"
        return [EdiQSONord(r) for r in self.db.sqlselect(sql, (self.contestname, band))]


def edi_export(db, contestname, edi_template, exp_prefix, editype):

    if os.path.isabs(edi_template):
        template = edi_template
    else:
        template = os.path.join(get_conf_directory(), os.path.basename(edi_template))
    if editype == EDITYPE.EDI_NORD:
        edi = EdiExporterNordContest(db, contestname, template, exp_prefix)
    else:
        edi = EdiExporter(db, contestname, template, exp_prefix)
    edi.export()
