#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
7log: cmd line logbook
written by dg7bbp

Initially developed for gqrx_set_trx.py

dxcluster access

(c) 2019-2023, dg7bbp, Jens Rosebrock
"""
import socket
import datetime
import time
from xmlrpc.server import SimpleXMLRPCServer, SimpleXMLRPCRequestHandler
import xmlrpc.client
import threading


# Restrict to a particular path.
class RequestHandler(SimpleXMLRPCRequestHandler):
    rpc_paths = ("/RPC2",)


class DxCluster(object):
    def __init__(self, server, port, login):
        self.xmlrpcport = 9177
        self.encoding = "iso8859_15"
        self.server = server
        self.port = port
        self._login = login
        self.readstream = None
        self.writestream = None
        self.sock = None
        self.readyfordx = False
        self.xmlserver = None

    def startxmlrpcserver(self):
        """
        starts a thread xmplrpcserver for spoting message
        """
        self.serverthread = threading.Thread(target=self._xmlserver)
        self.serverthread.start()

    def _xmlserver(self):
        with SimpleXMLRPCServer(
            ("localhost", self.xmlrpcport), requestHandler=RequestHandler
        ) as server:
            self.xmlserver = server
            server.register_introspection_functions()
            server.register_function(self._spot_qso)
            server.serve_forever()

    def xml_rpc_send_qso(self, frequency, call, message):
        s = xmlrpc.client.ServerProxy("http://localhost:{}".format(self.xmlrpcport))
        try:
            return s._spot_qso(frequency, call, message)
        except ConnectionRefusedError:
            # ohne grabbing direkt senden.
            # aber vorher eine Verbindung aufbauen
            if not self.sock:
                self.connect()
                self.loginDX()
            return self._spot_qso(frequency, call, message)

    def shutdown_xml(self):
        if self.xmlserver is not None:
            self.xmlserver.shutdown()
            self.xmlserver = None

    def connect(self, localfile=None):
        """
        connects to server
        """
        if localfile:
            self.readstream = open(localfile, "r")
        else:
            self.sock = socket.create_connection((self.server, self.port), timeout=None)
            self.readstream = self.sock.makefile(
                encoding=self.encoding, errors="backslashreplace"
            )
            self.writestream = True
            # self.writestream = self.sock.makefile(mode="w", encoding="utf-8")

    def __enter__(self):
        """
        Allow with statement
        """
        self.connect()
        self.loginDX()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.disconnect()

    def disconnect(self):
        if self.readstream is not None:
            self.readstream.close()
        if self.writestream is not None:
            self.writestream = False

        if self.sock is not None:
            self.sock.close()

    def loginDX(self):
        if self.writestream is not None:
            self.sock.send("{}\r\n".format(self._login).encode(self.encoding))
            l = self.readstream.readline()
            self.readyfordx = True
        else:
            self.readyfordx = True
            return True

    def read_dx(self):
        if self.readyfordx:
            has_dx = False
            while True:
                l = self.readstream.readline()
                if l:
                    line = l.strip()
                    if line.startswith("DX de"):
                        print(line)
                        dx_message = self._splitmessage(line)
                        if dx_message is not None:
                            has_dx = True
                            yield dx_message
                        else:
                            print("INVALID DX line")
                    else:
                        # show login messages
                        # no oter announcements
                        if not has_dx:
                            print(line)

    def _splitmessage(self, line):
        callend = line[6:].find(":")
        if callend > 0:
            srccall = line[6 : callend + 6]
            frequstr = line[callend + 7 : 25].strip()
            freq = float(frequstr) * 1000.0
            callend = line[26:].find(" ")
            heardcall = ""
            if callend > 3:
                heardcall = line[26 : 26 + callend]
            comment = line[27 + callend : 75]
            if len(line) > 77:
                location = line[76:80]
            else:
                location = ""
            date = datetime.datetime.utcnow()
            return {
                "srccall": srccall,
                "heardcall": heardcall,
                "frequency": freq,
                "comment": comment.strip(),
                "spotdate": date.isoformat(),
                "location": location,
            }
        return None

    def spot_qso(self, frequency, call, message):
        return self.xml_rpc_send_qso(frequency, call, message)
        # on exception use direct call

    def _spot_qso(self, frequency, call, message):
        fstr = "{:0.1f}".format(frequency / 1000.0)
        m = "DX {} {} {}\n".format(fstr, call, message)
        self.sock.send(m.encode(self.encoding))
        time.sleep(2)
        return m


def test():
    dx = DxCluster("dx-cluster.de", 4000, "dg7bbp")
    # dx.connect("cluster.txt")
    dx.connect()
    dx.loginDX()
    for dx in dx.read_dx():
        print(dx)
    dx.disconnect()


if __name__ == "__main__":
    test()
