#! /usr/bin/env python3
# -*- coding: utf-8 -*-

import collections
import re
import os
import datetime
import requests
import tempfile
import tarfile
import shutil


class CountryDesc(object):
    def __init__(
        self, name, continent, utc, lat, lon, ituzone, cqzone, adif, deleted, dates
    ):
        self.name = name
        self.continent = continent
        self.utc = utc
        self.lat = None
        self.lon = None
        try:
            if lat.endswith("N"):
                self.lat = float(lat[0:-1])
            elif lat.endswith("S"):
                self.lat = -float(lat[0:-1])
            if lon.endswith("E"):
                self.lon = float(lon[0:-1])
            elif lon.endswith("W"):
                self.lon = -float(lon[0:-1])
        except ValueError:
            pass
        self.ituzone = ituzone
        self.cqzone = cqzone
        self.adif = adif
        self.deleted = deleted
        self.valid_from = None
        self.valid_to = None
        if dates:
            splitted = dates.split("=")
            if len(splitted) > 1:
                datepart = splitted[0]
                if datepart:
                    split2 = datepart.split("-")
                    if len(split2) > 1 and split2[1]:
                        self.valid_to = self.todate(split2[1])
                    if len(split2) > 0 and split2[0]:
                        self.valid_from = self.todate(split2[0])

    def todate(self, datestr):
        splitted = datestr.split("/")
        if len(splitted) > 2 and splitted[0].isdigit():
            return datetime.date(int(splitted[0]), int(splitted[1]), int(splitted[2]))

    def is_valid(self):
        valid = True
        now = datetime.date.today()
        if self.valid_from is not None:
            valid = now >= self.valid_from
        if valid and self.valid_to is not None:
            valid = self.valid_to >= now
        return valid

    def __str__(self):
        return "Country: {}, ITU: {}, cqzone: {}".format(
            self.name, self.ituzone, self.cqzone
        )


class DXCCData(object):
    def __init__(self, filenames):
        self.filenames = filenames
        self.prefixes = list()  # (prefix,  CountryDesc)
        self.prefix_by_letter = collections.defaultdict(list)
        self.prefix_re_by_letter = dict()
        self.read_data()

    def prefix_to_reg_exp(self, prefix):
        """
        JA0[YZ]%% J[B-S]0[YZ]%% 7[J-N]0[YZ]%% 8[J-N]0[YZ]%%
        % is any asci char [A-Z]
        # is a single digit [0-9]
        [B-S] is like regexp
        """
        prefix = prefix.strip()
        has_reg = (
            prefix.find(r"%") >= 0 or prefix.find("#") > 0 or prefix.find("[") >= 0
        )
        prefix = prefix.replace(r"%", "[A-Z]")
        prefix = prefix.replace("#", "[0-9]")
        if has_reg:
            prefix += "$"
        else:
            prefix = "^" + prefix + "[0-9]{0,2}[A-Z]*"
        prefix_re = re.compile(prefix, re.I | re.A)
        return prefix_re

    def read_data(self):
        """
        read data from file
        splitted by |.1st field splitted by space prefixes
        """
        for filename in self.filenames:
            if os.path.isfile(filename):
                with open(filename, "r") as f:
                    for i, l in enumerate(f.readlines()):
                        line = l.strip().split("|")
                        if len(line) >= 7:
                            country = CountryDesc(*line[1:11])
                            if country.is_valid():
                                prefixes = line[0].split(" ")
                                for prefix in prefixes:
                                    stripped_prefix = prefix.strip()
                                    if stripped_prefix:
                                        self.prefix_by_letter[
                                            stripped_prefix[0].upper()
                                        ].append((prefix, country))
                                        # prefix_re = self.prefix_to_reg_exp(prefix)
                                        # self.prefixes.append((prefix_re, prefix, country))

    def _prefix_re_by_letter(self, call_letter):
        """ """
        call_letter = call_letter.upper()
        res_by_letter = self.prefix_re_by_letter.get(call_letter)
        if res_by_letter is None:
            pref_for_letter = []
            for prefix, country in self.prefix_by_letter[call_letter]:
                prefix_re = self.prefix_to_reg_exp(prefix)
                pref_for_letter.append((prefix_re, country))
                res_by_letter = pref_for_letter
                self.prefix_re_by_letter[call_letter] = pref_for_letter
        return res_by_letter

    def call_data(self, call):
        """
        strips prefixes and suffixes from call
        """
        call = call.upper().strip()
        splitted_call = call.split("/")
        if splitted_call[-1] in ["M", "MM", "AM", "P"]:
            splitted_call = splitted_call[0:-1]
        if len(splitted_call) > 2:
            # we have an extension and an unknown prefix
            # remove the prefix
            splitted_call = splitted_call[1:]
        # try complete call or every single element
        country_data = self._call_data("/".join(splitted_call))
        if country_data is None:
            for call_part in splitted_call:
                country_data = self._call_data(call_part)
                if country_data is not None:
                    break
        return country_data

    def _call_data(self, call):
        """
        param call: str call without prefix
        returns
        """
        retCountry = None
        if call:
            call = call.upper()
            prefixes_re = self._prefix_re_by_letter(call[0])
            if prefixes_re:
                for pref, country in prefixes_re:
                    if pref.match(call):
                        retCountry = country
                        break
        return retCountry


def test():
    import time

    t1 = time.time()
    dxcc = DXCCData(["./AreaOK1RR.tbl", "./Country.tab"])
    t2 = time.time()
    print(dxcc.call_data("dl2ab"))
    print(dxcc.call_data("dl2abc"))
    t3 = time.time()
    print(t3 - t2)
    print(dxcc.call_data("dg7bbp"))
    t3a = time.time()
    print(t3a - t3)

    print(dxcc.call_data("9k2jj"))
    print(dxcc.call_data("f4krc"))
    print(dxcc.call_data("ok1krc"))
    print(dxcc.call_data("by3f"))
    t4 = time.time()
    print(t2 - t1)
    print(t4 - t3a)
    print(dxcc.call_data("dl/by3f/p"))
    print(dxcc.call_data("ik5/dg7bbp"))
    print(dxcc.call_data("c31ab"))


def testUpdater():
    au = AreaUpdater()
    au.updateAreaDb("/tmp")


class AreaUpdater:

    to_extract = ["./AreaOK1RR.tbl", "./Country.tab"]

    def __init__(self):
        pass

    def updateAreaDb(self, dst_path):
        tgz_file = self.download()
        if tgz_file:
            try:
                tmp_dir = tempfile.mkdtemp()
                self.extract(tgz_file, tmp_dir)
                self.copy_to_config(tmp_dir, dst_path)
            finally:
                shutil.rmtree(tmp_dir)
                os.unlink(tgz_file)

    def download(self):
        tgzname = None
        r = requests.get(
            "http://www.ok2cqr.com/linux/cqrlog/ctyfiles/cqrlog-cty.tar.gz", stream=True
        )
        if r.status_code == 200:
            with tempfile.NamedTemporaryFile(suffix=".tgz", delete=False) as df:
                for data in r.iter_content(4096):
                    df.write(data)
                tgzname = df.name
        return tgzname

    def extract(self, tgzname, dst_path):
        with tarfile.TarFile.open(tgzname, mode="r|gz") as tf:
            for tm in tf:
                if tm.isfile() and tm.name in self.to_extract and tm.size < 1e6:
                    tf.extract(tm, set_attrs=False, path=dst_path)

    def copy_to_config(self, src_path, dst_path):
        for fname in self.to_extract:
            src_file = os.path.normpath(os.path.join(src_path, fname))
            dst_file = os.path.normpath(os.path.join(dst_path, fname))
            if os.path.isfile(src_file) and os.path.getsize(src_file) > 1024:
                shutil.copy2(src_file, dst_file)
            if os.path.isfile(src_path):
                os.unlink(src_path)


if __name__ == "__main__":
    testUpdater()
    test()
