#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019-2022 dg7bbp, Jens Rosebrock

DB Wrapper or sqlite
"""
import json
import os
import sqlite3
import time
from logconfig import ContestDesc
from pygments.lexers._postgres_builtins import PLPGSQL_KEYWORDS


class LogDbAccess(object):

    tables = {
        "SESSION": {
            "columns": [
                ("sessionid", "TEXT", False, False),
                ("current_row", "INTEGER", False, False),
                ("last_row", "INTEGER", False, False),
                ("active_contest", "TEXT", False, False),
            ],
            "index": {},
        },
        "QSOS": {
            "columns": [
                ("src_call", "TEXT", False, False),
                ("operator_call", "TEXT", False, False),
                ("src_locator", "TEXT", False, False),
                ("dst_call", "TEXT", False, False),
                ("dst_operator", "TEXT", False, False),
                ("rst_send", "TEXT", False, False),
                ("rst_received", "TEXT", False, False),
                ("repeater", "TEXT", False, False),
                ("freq", "REAL", False, False),
                ("freq_rx", "REAL", False, False),
                ("mode", "TEXT", False, False),
                ("prop", "TEXT", False, False),
                ("band", "TEXT", False, False),
                ("band_rx", "TEXT", False, False),
                ("dst_locator", "TEXT", False, False),
                ("begin_date", "TEXT", False, False),
                ("begin_time", "TEXT", False, False),
                ("end_time", "TEXT", False, False),
                ("name", "TEXT", False, False),
                ("comment", "TEXT", False, False),
                ("qslcomment", "TEXT", False, False),
                ("qslreq", "INTEGER", False, False),
                ("qslin", "INTEGER", False, False),
                ("qslout", "INTEGER", False, False),
                ("rig", "TEXT", False, False),
                ("antenna", "TEXT", False, False),
                ("contest_name", "TEXT", False, False),
                ("manager", "TEXT", False, False),
                ("contest_ex_tx", "TEXT", False, False),
                ("contest_ex_rx", "TEXT", False, False),
                ("continent", "TEXT", False, False),
                ("dst_cqzone", "TEXT", False, False),
                ("dst_ituzone", "TEXT", False, False),
            ],
            "index": {},
        },
        "markers": {
            "columns": [
                ("call", "TEXT", True, True),
                ("band", "TEXT", True, True),
                ("frequency", "REAL", True, False),
                ("timestamp", "TEXT", True, False),
                ("qtf", "REAL", False, False),
            ],
            "index": {},
        },
        "qrzcache": {
            "columns": [
                ("call", "TEXT", True, True),
                ("m_date", "REAL", True, False),
                ("info", "TEXT", False, False),
            ]
        },
        "dxcluster": {
            "columns": [
                ("srccall", "TEXT", False, False),
                ("heardcall", "TEXT", True, True),
                ("frequency", "REAL", True, True),
                ("band", "TEXT", True, True),
                ("spotdate", "TEXT", True, True),
                ("location", "TEXT", False, False),
                ("comment", "TEXT", False, False),
            ]
        },
        "contests": {
            "columns": [
                ("contest_name", "TEXT", True, True),
                ("number_per_band", "INTEGER", True, False),
                ("number_scheme", "TEXT", True, False),
                ("exchange_data", "TEXT", True, False),
                ("required_fields_attrs", "TEXT", True, False),
                ("unique_station_attrs", "TEXT", True, False),
            ],
            "index": {},
        },
        "contest_numbers": {
            "columns": [
                ("contest_name", "TEXT", True, True),
                ("band", "TEXT", True, True),
                ("number", "INTEGER", True, False),
            ],
            "index": {},
        },
        "qslexportjobs": {
            "columns": [
                ("exporttime", "TEXT", True, False),
                ("jobcommitted", "INTEGER", False, False),
            ],
            "index": {},
        },
        "qslexported": {
            "columns": [
                ("exportjob", "INTEGER", True, True),
                ("qsonumber", "INTEGER", True, True),
            ],
            "index": {},
        },
    }

    def __init__(self, db_filename, session_id):
        self.db_filename = db_filename
        self.session_id = session_id
        self.conn = None
        self._data_attributes = dict()
        self._create_or_update_db(db_filename)

    def _init_db_tables(self):
        """
        inits or update db
        """
        c = self.conn.cursor()
        c.execute("select name, sql from sqlite_master where type = 'table'")
        name_to_sql = dict()
        for r in c:
            name_to_sql[r[0]] = r[1]
        index_to_sql = dict()
        c.execute("select name, sql from sqlite_master where type = 'index'")
        for r in c:
            index_to_sql[r[0]] = r[1]
        for t_name, t_info in self.tables.items():
            existing_table = name_to_sql.get(t_name)
            if not existing_table:
                self._create_table(c, t_name, t_info)
            else:
                self._update_if_necessary(c, t_name, t_info, existing_table)
            index_names = t_info.get("index", dict())
            for index_name, index_attrs in index_names.items():
                existing_index = index_to_sql.get(index_name)
                if not existing_index:
                    self._create_db_index(c, t_name, index_name, index_attrs)
                else:
                    pass
                    # update for index do it later
        self.conn.commit()

    def _create_table(self, cursor, t_name, t_info):
        """
        Creates table in database
        """
        primary_key = []
        columns = []
        self._data_attributes[t_name] = list()
        for col_info in t_info["columns"]:
            colname, coltype, not_null, primary = col_info
            self._data_attributes[t_name].append(colname)
            if primary:
                primary_key.append(colname)
            col_desc = [colname, coltype]
            if not_null:
                col_desc.append("NOT NULL")
            col_desc_str = " ".join(col_desc)
            columns.append(col_desc_str)
        prim_key = ""
        if primary_key:
            prim_key = "PRIMARY KEY (%s)" % ", ".join(primary_key)
            columns.append(prim_key)
        sql = "create table %s (%s)" % (t_name, ", ".join(columns))
        cursor.execute(sql)

    def _create_db_index(self, cursor, t_name, index_name, index_attributes):
        """
        Create index in database
        """
        sql = "create index %s on %s (%s)" % (
            index_name,
            t_name,
            ", ".join(index_attributes),
        )
        cursor.execute(sql)

    def _update_if_necessary(self, cursor, t_name, t_info, existing_table):
        """
        very simple update
        """
        begin = existing_table.find("(")
        fields_str = existing_table[begin + 1 :]
        end = fields_str.find("PRIMARY KEY (")
        if end == -1:
            end = fields_str.find(")")
        field_str = fields_str[:end]
        fields = field_str.split(",")
        existing_names = dict()
        for f in fields:
            expr = f.strip()
            if expr:
                names = expr.split(" ")
                if names:
                    existing_names[names[0]] = names[1]
        # only add missing
        for col_info in t_info["columns"]:
            colname, coltype, not_null, _primary = col_info
            if colname not in existing_names:
                nn = ""
                if not_null:
                    if coltype == "INTEGER":
                        def_val = 0
                    else:
                        def_val = "''"
                    nn = "DEFAULT %s NOT NULL" % def_val
                sql = "ALTER table %s ADD %s %s %s" % (t_name, colname, coltype, nn)
                cursor.execute(sql)

    def _create_or_update_db(self, db_filename):
        """
        creates database and tables if needed updates tables
        returns connection.
        Set Return values of execute to sqlite.Row (not only tuples
        current_row = -1 when commit
        last_row = -1 if not exists (1st entry)
        last_row is updated on commit
        """
        db_dir = os.path.dirname(db_filename)
        if not os.path.isdir(db_dir):
            os.makedirs(db_dir, mode=0o755)

        conn = sqlite3.connect(db_filename)
        self.conn = conn
        self._init_db_tables()
        conn.row_factory = sqlite3.Row

    def _copy_qsos(self):
        """
        removes wrong attributes from a given qso table
        """
        cur = self.conn.cursor()
        columns = []
        cdefs = self.tables["QSOS_TMP"]["columns"]
        for c in cdefs:
            columns.append(c[0])
        cList = ", ".join(columns)
        sql = "insert into QSOS_TMP ({}) select {} from qsos".format(cList, cList)
        cur.execute(sql)
        self.conn.commit()
        cur.close()

    def insert_qso(self, data):
        """
        commits last session if needed
        and insert qso
        """
        session = self._get_session()
        if session and self.is_uncommited():
            self.commit_session()
            session = self._get_session()
        attrs = list()
        values = list()
        placeholders = list()
        for k, v in data.items():
            attrs.append(k)
            values.append(v)
            placeholders.append("?")
        attrs_s = ", ".join(attrs)
        placeh_s = ", ".join(placeholders)
        sql = "insert into qsos (%s) values (%s)" % (attrs_s, placeh_s)
        cur = self.conn.cursor()
        cur.execute(sql, tuple(values))
        current_row = cur.lastrowid
        last_row = -1
        if session and session["last_row"] >= 0:
            last_row = session["last_row"]
            self.update_session(cur, current_row, last_row)
        else:
            self.add_session(cur, current_row)
        cur.close()
        self.conn.commit()

    def addIfNotExists(self, data):
        """
        look up record by call, date, time, band
        if not exists,inserts record
        assumes complete data
        """
        imported = False
        sql = (
            "select _rowid_  from qsos where src_call =? and dst_call =? "
            " and begin_date=? and begin_time=? and band=?"
        )
        cur = self.conn.cursor()
        cur.execute(
            sql,
            (
                data["src_call"],
                data["dst_call"],
                data["begin_date"],
                data["begin_time"],
                data["band"],
            ),
        )

        exists = cur.fetchone() is not None
        cur.close()
        if not exists:
            imported = True
            self.insert_qso(data)
        return imported

    def update_qso(self, data, qso_number=None):
        if qso_number is None:
            row = -2
            session = self._get_session()
            if session is not None:
                row = session["current_row"]
        else:
            row = qso_number
        if row > 0:
            cur = self.conn.cursor()
            attrs = []
            values = []
            for key, value in data.items():
                attrs.append("%s=?" % key)
                values.append(value)

            attr_str = ", ".join(attrs)
            if attr_str:
                sql = "update qsos set %s where _rowid_=?" % attr_str
                values.append(row)
                cur.execute(sql, tuple(values))
            cur.close()
            self.conn.commit()

    def commit_session(self):
        if self.is_uncommited():
            session = self._get_session()
            cur = self.conn.cursor()
            self.update_session(cur, -1, session["current_row"])
            self.conn.commit()

    def cancel_session(self):

        if self.is_uncommited():
            session = self._get_session()
            cur = self.conn.cursor()
            self.delete_qso(cur, session["current_row"])
            self.update_session(cur, -1, session["last_row"])
            self.conn.commit()

    def is_uncommited(self):
        uncommitted = False
        session = self._get_session()
        if session and session["current_row"] != -1:
            uncommitted = True
        return uncommitted

    def can_commit(self):
        """
        :returns list of missing values or None if no current record
        """
        missing_values = None
        needed_vals = ["src_call", "dst_call", "band", "mode"]
        session = self._get_session()
        if session and session["current_row"] != -1:
            missing_values = []
            qso = self.get_qso_by_number(session["current_row"])
            for k in needed_vals:
                if not qso[k]:
                    missing_values.append(k)
        return missing_values

    def current_call(self):
        call = None
        session = self._get_session()
        if session is not None:
            current_row = session["current_row"]
            if current_row >= 0:
                qso = self.get_qso_by_number(current_row)
                if qso is not None:
                    call = qso["dst_call"]
        return call

    def current_qso(self):
        """
        :returns row of current active qso or None
        """
        qso = None
        session = self._get_session()
        if session is not None:
            current_row = session["current_row"]
            if current_row >= 0:
                qso = self.get_qso_by_number(current_row)
        return qso

    def last_qso(self):
        """
        returns values of last qso, to get mode band etc.
        from last qso if no hamlib is available
        """
        qso_row = None
        session = self._get_session()
        if session is not None:
            last_qso = session["last_row"]
            if last_qso >= 0:
                qso_row = self.get_qso_by_number(last_qso)
        return qso_row

    def last_qso_number(self):
        """
        :returns last_qso number
        """
        session = self._get_session()
        last_qso = None
        if session is not None:
            last_qso = session["last_row"]
        return last_qso

    def last_or_current(self):
        """
        returns values of last qso, to get mode band etc.
        from last qso if no hamlib is available
        """
        qso_row = None

        session = self._get_session()
        if session is not None:
            if self.is_uncommited():
                last_qso = session["current_row"]
            else:
                last_qso = session["last_row"]
            if last_qso >= 0:
                qso_row = self.get_qso_by_number(last_qso)
        return qso_row

    def get_last_qsos(self, call, number):
        cur = self.conn.cursor()
        row_id = -1
        session = self._get_session()
        if session is not None:
            row_id = session["current_row"]

        cur.execute(
            "select * from qsos where dst_call=? and _rowid_ <> ? order by begin_date desc, begin_time desc",
            (call, row_id),
        )
        for i in range(number):
            row = cur.fetchone()
            if row is None:
                cur.close()
                break
            else:
                yield row

    def close(self):
        self.conn.close()

    def _get_session(self):
        """ """
        cur = self.conn.cursor()
        cur.execute("select *  from session where sessionid=?", (self.session_id,))
        row = cur.fetchone()
        cur.close()
        return row

    def update_session(self, cur, current_row, last_row):
        cur.execute(
            "update session set current_row=?, last_row=? where sessionid=?",
            (current_row, last_row, self.session_id),
        )

    def add_session(self, cur, current_row):
        cur.execute(
            "insert into session (sessionid, current_row, last_row) "
            " values (?,?, -1)",
            (self.session_id, current_row),
        )

    def get_qso_by_number(self, number):
        cur = self.conn.cursor()
        cur.execute("select *  from qsos where _rowid_= ?", (number,))
        row = cur.fetchone()
        cur.close()
        return row

    def delete(self, rownumber):
        session = self._get_session()
        if session is not None:
            current_row = session["current_row"]
            if current_row == rownumber:
                print("Not allowed to delete current qso")
                return
        cur = self.conn.cursor()
        self.delete_qso(cur, rownumber)
        self.conn.commit()

    def delete_qso(self, cur, rownumber):
        cur.execute("delete from qsos where _rowid_ = ?", (rownumber,))

    def get_qsos(self, from_date, to_date, src_call):
        cur = self.conn.cursor()
        if to_date and to_date.lower() != "now":
            if src_call is not None:
                cur.execute(
                    "select _rowid_, * from qsos where src_call =? and (begin_date >=? and begin_date <=?) order by begin_date asc, begin_time asc",
                    (src_call, from_date, to_date),
                )
            else:
                cur.execute(
                    "select _rowid_, * from qsos where begin_date >=? and begin_date <=? order by begin_date asc, begin_time asc",
                    (from_date, to_date),
                )
        else:
            if src_call is not None:
                cur.execute(
                    "select _rowid_, * from qsos where src_call=? and begin_date >=? order by begin_date asc, begin_time asc",
                    (src_call, from_date),
                )
            else:
                cur.execute(
                    "select _rowid_, * from qsos where begin_date >=? order by begin_date asc, begin_time asc",
                    (from_date,),
                )
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def request_qsl_cards(self, src_call, loc_filter):

        cur = self.conn.cursor()
        # get all qso with request and not marked as out
        qsl_filter = " and (upper(qslreq) = 'QSLREQ' and (not upper(qslout) == 'QSLOUT' or qslout is NULL)) "
        if loc_filter:
            cur.execute(
                "select _rowid_, * from qsos where src_call=? and src_locator=?"
                + qsl_filter,
                (src_call, loc_filter),
            )
        else:
            cur.execute(
                "select _rowid_, * from qsos where src_call=?" + qsl_filter, (src_call,)
            )
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def search_qsos(self, s):
        s_fields = ["dst_call", "name", "comment", "dst_locator"]
        s_cond = " or ".join(["{} like ?".format(x) for x in s_fields])
        sql = "select _rowid_, * from qsos where + {} order by begin_date asc, begin_time asc".format(
            s_cond
        )
        s_string = "%%{}%%".format(s)
        s_list = [s_string] * len(s_fields)
        cur = self.conn.cursor()
        cur.execute(sql, tuple(s_list))
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def last_qsos(self, max_number):
        sql = "select _rowid_, * from qsos order by _rowid_ desc limit ?"
        cur = self.conn.cursor()
        cur.execute(sql, (max_number,))
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def update_or_insert_marker(self, call, band, frequency, qtf, timestamp):
        """
        Update marker table
        """
        cur = self.conn.cursor()
        sql = "insert or replace into markers (call, band, frequency, qtf, timestamp) values (?,?,?,?,?)"
        cur.execute(sql, (call, band, frequency, qtf, timestamp))
        cur.close()
        self.conn.commit()

    def list_markers(self):
        cur = self.conn.cursor()
        sql = "select call, band, frequency, qtf, timestamp from markers order by frequency"
        cur.execute(sql)
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def delete_markers(self):
        """
        delete markes, deletiton must be complete by self.conn.commit or rollback
        """
        cur = self.conn.cursor()
        sql = "delete from markers"
        cur.execute(sql)
        n = cur.rowcount
        cur.close()
        # self.conn.commit() # think about outside and commit or rollback later
        return n

    def find_marker_by_call(self, call):
        cur = self.conn.cursor()
        sql = "select call, frequency, qtf, timestamp from markers  where call like ? order by frequency"
        cur.execute(sql, ("%%%s%%" % call,))
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def find_marker_by_frequency(self, f, epsilon=1e3):
        cur = self.conn.cursor()
        sql = "select call, frequency, qtf, timestamp from markers where frequency>=? and frequency<=? order by frequency"
        cur.execute(sql, (f - epsilon, f + epsilon))
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()

    def update_qrz_cache(self, call, info):
        """
        updates qrz cache forcall with dictinfo from info
        """
        cur = self.conn.cursor()
        sql = "insert or replace into qrzcache (call, m_date, info) values (?,?,?)"
        cur.execute(sql, (call.upper(), time.time(), json.dumps(info)))
        cur.close()
        self.conn.commit()

    def lookup_qrz_cache(self, call):
        """
        lookup for call in qrz_cache
        returns info, modified_date
        """
        cur = self.conn.cursor()
        sql = "select m_date, info from qrzcache where call=?"
        cur.execute(sql, (call.upper(),))
        row = cur.fetchone()
        info = None
        m_date = None
        if row is not None:
            info = json.loads(row["info"])
            m_date = row["m_date"]
        return info, m_date

    def known_contests(self):
        """
        returns all know contests
        """
        cur = self.conn.cursor()
        sql = "select * from contests order by contest_name"
        cur.execute(sql)
        row = cur.fetchone()
        while row is not None:
            yield dict(row)
            row = cur.fetchone()

    def contest_info(self, contest_name):
        """
        returns ContestDesc for contest_name or None
        """
        c_info = None
        cur = self.conn.cursor()
        sql = "select * from contests where contest_name=?"
        cur.execute(sql, (contest_name,))
        row = cur.fetchone()
        if row is not None:
            c_info = ContestDesc(
                row["contest_name"],
                {
                    "number_per_band": row["number_per_band"] == 1,
                    "number_scheme": row["number_scheme"],
                    "required_attrs": json.loads(row["required_fields_attrs"]),
                    "unique_station_attrs": json.loads(row["unique_station_attrs"]),
                },
            )
        return c_info

    def create_contest(
        self,
        contest_name,
        number_per_band,
        number_scheme,
        exchange_data,
        required_fields,
        unique_station_attrs,
    ):
        """[("contest_name", "TEXT", True, True),
        ("number_per_band", "INTEGER", True, False),
        ("number_scheme", "TEXT", True, False),
        ("required_fields_attrs", "TEXT", True, False),
        ("unique_station_attrs", "TEXT", True, False)],
        """
        cur = self.conn.cursor()
        sql = (
            "insert or replace into contests (contest_name, number_per_band, "
            "number_scheme, exchange_data, required_fields_attrs, unique_station_attrs) "
            "values (?, ?, ?, ?, ?, ?)"
        )
        cur.execute(
            sql,
            (
                contest_name,
                1 if number_per_band else 0,
                number_scheme,
                exchange_data,
                json.dumps(required_fields),
                json.dumps(unique_station_attrs),
            ),
        )
        # number table is initilized on request,
        # because we don't know the used bands
        cur.close()
        self.conn.commit()

    def update_contest(
        self,
        contest_name,
        number_per_band,
        number_scheme,
        required_fields,
        unique_station_attrs,
    ):
        """[("contest_name", "TEXT", True, True),
        ("number_per_band", "INTEGER", True, False),
        ("number_scheme", "TEXT", True, False),
        ("required_fields_attrs", "TEXT", True, False),
        ("unique_station_attrs", "TEXT", True, False)],
        """
        cur = self.conn.cursor()
        sql = (
            "update contests set number_per_band=?, "
            "number_scheme=?, required_fields_attrs=?, unique_station_attrs=? "
            "where contest_name=?"
        )
        cur.execute(
            sql,
            (
                1 if number_per_band else 0,
                number_scheme,
                json.dumps(required_fields),
                json.dumps(unique_station_attrs),
                contest_name,
            ),
        )
        # number table is initilized on request,
        # because we don't know the used bands
        cur.close()
        self.conn.commit()

    def update_contest_number(self, contest_name, band):
        """
        returns next number
        without commit db!. Need a transaction for storing qso and contestnumber
        """
        number = self.get_contest_number(contest_name, band)
        cur = self.conn.cursor()
        if number == 0:
            # insert
            number = 1
            sql = (
                "insert into contest_numbers (contest_name, band, number) values(?,?,?)"
            )
            cur.execute(sql, (contest_name, band, number))
        else:
            # update
            number += 1
            sql = "update contest_numbers set number=? where contest_name=? and band=?"
            cur.execute(sql, (number, contest_name, band))
        cur.close()
        self.conn.commit()
        return number

    def get_contest_number(self, contest_name, band, cursor=None):
        """
        band maybe ALL oder bandname
        returns current contest_number 0 if not started
        """
        number = 0
        if cursor is None:
            local_cursor = self.conn.cursor()
        else:
            local_cursor = cursor
        sql = "select number from contest_numbers where contest_name=? and band=?"
        local_cursor.execute(sql, (contest_name, band))
        row = local_cursor.fetchone()
        if row is not None:
            number = row["number"]
        if cursor is None:
            local_cursor.close()
        return number

    def contest_numbers(self, contest_name):
        """
        yields (band, curr_number)
        """
        cur = self.conn.cursor()
        sql = "select number, band from contest_numbers where contest_name=? order by band"
        cur.execute(sql, (contest_name,))
        row = cur.fetchone()
        while row is not None:
            yield row["band"], row["number"]
            row = cur.fetchone()
        cur.close()

    def set_contest_mode(self, contest_name):
        """
        sets current session into contestmode
        contest_name must exist or must be empty.
        """
        cur = self.conn.cursor()
        session = self._get_session()
        if session is None:
            self.add_session(cur, -1, -1)
        sql = "update session set active_contest=? where sessionid=?"
        cur.execute(sql, (contest_name, self.session_id))
        self.conn.commit()

    def current_contest(self):
        """
        returns current contestname
        """
        contest_name = ""
        session = self._get_session()
        if session is not None:
            contest_name = session["active_contest"]
        return contest_name

    def contest_double_check(self, curr_contest, unique_vals):
        """
        curr_contest: contestname
        unique_val:key_value pairs to check for contest
        """
        ret = None
        cur = self.conn.cursor()
        conds = []
        for attr in unique_vals.keys():
            conds.append("%s=:%s" % (attr, attr))
        unique_vals["contest_name"] = curr_contest
        sql = (
            "select _rowid_, * from qsos where contest_name=:contest_name and  %s"
            % " AND  ".join(conds)
        )
        cur.execute(sql, unique_vals)
        row = cur.fetchone()
        if row is not None:
            ret = dict(row)
        cur.close()
        return ret

    def insert_dxcluster_spot(self, spotdict, remove_date):
        """
        insert spotdict into dxcluster
        """
        cur = self.conn.cursor()
        if remove_date is not None:
            sql = "delete from dxcluster where spotdate < :remove_date"
            cur.execute(sql, {"remove_date": remove_date})
        sql = (
            "insert or replace into dxcluster (srccall, heardcall, comment, frequency, band, spotdate, location) "
            "VALUES (:srccall, :heardcall, :comment, :frequency, :band, :spotdate, :location)"
        )
        cur.execute(sql, spotdict)
        cur.close()
        self.conn.commit()

    def get_dx_cluster_calls(self, frequency):
        sql = "select * from dxcluster where frequency>? and frequency <?"
        return self.sqlselect(sql, (frequency - 500.0, frequency + 500.0))

    def get_dx_cluster_spots(self, band):
        sql = "select * from dxcluster where band=? order by spotdate"
        return self.sqlselect(sql, (band,))

    def get_dx_cluster_call(self, call, bandfilter):
        if bandfilter is not None:
            sql = "select * from dxcluster where heardcall=? and band=? order by spotdate desc"
            return self.sqlselect(sql, (call, bandfilter))
        else:
            sql = "select * from dxcluster where heardcall=? order by spotdate desc"
            return self.sqlselect(sql, (call,))

    def sqlselect(self, sql, args):
        """
        runs select statement and yield rows as list of dicts
        """
        cur = self.conn.cursor()
        cur.execute(sql, args)
        row = cur.fetchone()
        while row is not None:
            yield dict(row)
            row = cur.fetchone()
        cur.close()

    def start_qsl_exportjobs(self, timestamp):
        """
        creates an export job with commit
        and returns rowid
        """
        cur = self.conn.cursor()
        sql = "insert into qslexportjobs (exporttime, jobcommitted ) VALUES (?, 0)"
        cur.execute(sql, (timestamp,))
        rowid = cur.lastrowid
        cur.close()
        return rowid

    def finish_qsl_export_job(self, rowid, qso_numbers, cancel):
        if cancel:
            self.conn.rollback()
        else:
            cur = self.conn.cursor()
            recs = [{"rowid": rowid, "qso": qso} for qso in qso_numbers]
            cur.executemany(
                "INSERT into qslexported (exportjob, qsonumber) values (:rowid, :qso)",
                recs,
            )
            cur.close()
            self.conn.commit()

    def qsl_export_done(self, row_id):
        """
        Updates qslout for transmitted adif files
        """
        cur = self.conn.cursor()
        sql = "update qsos set qslout='qslout' where _rowid_ in (select qsonumber from qslexported where exportjob=?)"
        cur.execute(sql, (row_id,))
        rows = cur.rowcount
        sqljobs = "update qslexportjobs set jobcommitted = 1 where _rowid_=?"
        cur.execute(sqljobs, (row_id,))
        cur.close()
        self.conn.commit()
        return rows

    def get_export_jobs(self):
        sql = "select _rowid_, * from qslexportjobs order by _rowid_"
        for r in self.sqlselect(sql, {}):
            yield r

    def search_qso_by_attrs(self, db_values):
        """
        search qso by give db_value k, v pairs
        using exact match only for dst_call uses like %call%
        that matched also portable qsos
        """
        s_fields = []
        values = []
        for k, v in db_values.items():
            s_fields.append(f"upper({k})")
            if k == "dst_call":
                v = f"%%{v}%%"
            values.append(v.upper())
        s_cond = " and ".join(["{} like ?".format(x, x) for x in s_fields])
        sql = "select _rowid_, * from qsos where + {} order by begin_date asc, begin_time asc".format(
            s_cond
        )
        cur = self.conn.cursor()
        cur.execute(sql, tuple(values))
        row = cur.fetchone()
        while row is not None:
            yield row
            row = cur.fetchone()
        cur.close()
