#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Part of
/log: cmd line logbook
written by dg7bbp

(c) 2019-2021, dg7bbp, Jens Rosebrock

Adif exporter
"""

import collections
import datetime
import io
import re
import sys

#
# AdifField = collections.namedtuple("AdifField", ["name", "fieldtype"])


def to_mhz(freq_in_hz, inv=False):
    if freq_in_hz is not None:
        if inv:
            return freq_in_hz * 1e6
        else:
            return freq_in_hz / 1e6
    else:
        return 0


class AdifField:
    def __init__(self, name, fieldtype, calc_func=None):
        """
        calc_func: callable to process value
        """
        self.name = name
        self.fieldtype = fieldtype
        self.calc_func = calc_func


class AdifType(object):
    Boolean = "B"
    Number = "N"
    Date = "D"
    Time = "T"
    String = "S"
    Enumeration = "E"
    Empty = None


class Adif(object):

    ADIF_VER = "3.0.4"

    def __init__(self, progid, call, db_to_adif, lotwmode=False):
        """
        :param db_to_adif dict: of db_name to AdifField
        """
        self.db_to_adif = db_to_adif
        self.progid = progid
        self.call = call
        self.file = None
        self.file_is_plain = True
        self._lotwmode = lotwmode

    def _is_sat(self, db_record):
        """
        check for sat name like qo-100 and set
        and add propagation mode
        :returns true if sat
        """
        is_sat = False
        repeater = db_record["repeater"]
        if repeater:
            is_sat = re.match("[a-z][a-z]-[0-9]+", repeater, re.I) != None
        return is_sat

    def calculated_val(self, db_record, db_key, adif_i):
        """
        ugly hack. Better uses classes instead of
        named tuple and generator to generate multilple
        fields if repeater is set in DB.
        """
        val = None
        if db_key == "__prop_mode":
            ## check for sat name like qo-100 and set
            ## and add propagation mode
            repeater = db_record["repeater"]
            if repeater:
                if self._is_sat(db_record):
                    val = "SAT"
                else:
                    val = "RPT"
        elif db_key == "__repeater":
            # only add satname if satmode propmode == SAT
            repeater = db_record["repeater"]
            if repeater:
                if self._is_sat(db_record):
                    val = repeater
        elif db_key == "__qsl_rcvd":
            val = ""
            qslout = db_record["qslout"]
            qslreq = db_record["qslreq"]
            qslin = db_record["qslin"]
            if qslin:
                val = "y"
            elif qslout:
                # should not happen,
                # only qsos without qslout
                val = None
            elif qslreq:
                val = "n"
        return val

    def append_record(self, db_record, to_file=True):
        """ """
        adif_fields = []
        for db_key, adif_i in self.db_to_adif.items():
            if db_key.startswith("__"):
                val = self.calculated_val(db_record, db_key, adif_i)
            else:
                val = db_record[db_key]
            if val:
                if adif_i.calc_func is not None:
                    val = adif_i.calc_func(val)
                s = self._format_value(adif_i.name, val, adif_i.fieldtype)
                ignore = adif_i.name == "prop_mode" and val == "RPT" and self._lotwmode
                if not ignore:
                    adif_fields.append(s)
        if to_file:
            line = "".join(adif_fields) + "<eor>\r\n"
            self.file.write(line)
        else:
            line = "".join(adif_fields) + "<eor>"
        return line

    def open_for_write(self, filename):
        if filename == "":
            self.file = sys.stdout
            self.file_is_plain = False
        else:
            self.file_is_plain = True
            self.file = io.open(filename, "w", encoding="UTF-8", newline="\n")
        self._write_header()

    def _write_header(self):
        dat = datetime.datetime.utcnow()
        d = dat.strftime("%Y-%m-%d")
        t = dat.strftime("%H:%M:%S")
        self.file.write("Generated on {} at {} UTC for {}\r\n".format(d, t, self.call))

        self.file.write(
            "{}\r\n".format(
                self._format_value("adif_ver", self.ADIF_VER, AdifType.Empty)
            )
        )
        self.file.write(
            "{}\r\n".format(
                self._format_value("programid", self.progid, AdifType.Empty)
            )
        )

        self.file.write("<EOH>\r\n\r\n")

    def _format_value(self, field_name, val_from_db, adif_type):
        if adif_type == AdifType.Number:
            val_from_db = f"{val_from_db:f}"
        adif_val = str(val_from_db)
        if adif_type == "D":
            # strip "-"
            adif_val = adif_val.replace("-", "")
        elif adif_type == "T":
            # strip ":"
            adif_val = adif_val.replace(":", "")
        if adif_type is not None and not self._lotwmode and adif_type == "S":
            s = "<{}:{}:{}>{}".format(field_name, len(adif_val), adif_type, adif_val)
        else:
            s = "<{}:{}>{}".format(field_name, len(adif_val), adif_val)
        return s

    def close(self):
        if self.file_is_plain:
            self.file.close()

    @staticmethod
    def readadif(fileh):
        """
        read adif file and returns list of dict with
        records.
        :param fileh: fileobject to read
        """
        state = 0
        # 0 beginning
        # 1 = after reading header or record
        # 2 = reading attrs

        qsolist = []
        for line in fileh:
            l = line.strip()
            if state == 0 and l.lower() == "<eoh>":
                state = 1
            elif state == 1:
                rec = dict()
                state = 2
            if state == 2 and l.startswith("<"):
                while l and state == 2:
                    mg = re.findall("<(.[_a-z]+):([0-9]+):[a-z]*>(.*)", l, re.I)
                    if not mg:
                        mg = re.findall("<(.[_a-z]+):([0-9]+)>(.*)", l, re.I)
                    if mg and len(mg[0]) == 3:
                        key = mg[0][0].lower()
                        data = mg[0][2]
                        vallen = int(mg[0][1])
                        val = data[0:vallen]
                        l = data[vallen:].strip()
                        rec[key] = val
                    elif l.lower() == "<eor>":
                        state = 1
                        qsolist.append(rec)
        return qsolist
