#! /usr/bin/env python3
# -*- coding: utf-8 -*-

"""
7log: cmd line logbook
written by dg7bbp

Initially developed for gqrx_set_trx.py

(c) 2019-2025, dg7bbp, Jens Rosebrock
"""

import argparse
import os
import sys
import re
import subprocess
import time
import datetime
import pathlib
from loglib import guess_args, frequency_to_band, time_defaults
from logconfig import read_config, set_config_value, read_rig, country_data_files
from logconfig import (
    rig_names,
    read_contest_templates,
    get_repeater_files,
    get_cloud_log,
    get_conf_directory,
)
from dbwrapper import LogDbAccess
from adif import Adif, AdifField, AdifType, to_mhz
from ediexport import edi_export, EDITYPE
from locator import bearing_degrees, distance_haversine, toLoc, Point
from qrzaccess import QrzAccess, QRZException
from hamlib import HamLibConnection
from repeater import Repeater
from dxcccqr import DXCCData, AreaUpdater
from dxcluster import DxCluster
from cloudlog import cloudlog_send_adif


_version = "1.1.9"


class ConfigException(Exception):
    pass


class BbpLog(object):

    _cmds = {
        "store",
        "cancel",
        "set",
        "cmdargs",
        "qsolist",
        "delete",
        "configuration",
        "listmarks",
        "delmarks",
        "findmarks",
        "map",
        "stations",
        "mark",
        "last",
        "contestfilter" "contestnew",
        "contestupdate",
        "contestlist",
        "contestinfo",
        "contestset",
        "riglist",
        "repeatersearch",
        "queryall",
        "edi",
        "edinord",
        "adifimport",
        "qrz",
        "who",
        "gotocall",
        "fetchareadata",
        "grabcluster",
        "spotqso",
        "printqso",
        "contest",
        "qslexport",
        "qsllistexport",
        "qslsend",
        "qslrecord",
    }

    _header = {
        "rowid": "#",
        "begin_date": "Date",
        "begin_time": "Time",
        "dst_call": "Call",
        "band": "Band",
        "mode": "Mode",
        "repeater": "via",
        "dst_locator": "Loc.",
        "rst_send": "RSTS",
        "context_ex_tx": "",
        "rst_received": "RSTR",
        "context_ex_rx": "",
        "name": "Name",
        "comment": "Comment",
        "qslin": "",
        "qslout": "",
        "qslreq": "",
        "manager": "",
    }

    _adif_db_map = {
        "dst_call": AdifField("call", AdifType.String),
        "band": AdifField("band", AdifType.Empty),
        "band_rx": AdifField("band_rx", AdifType.Empty),
        "freq": AdifField("freq", AdifType.Number, to_mhz),
        "freq_rx": AdifField("freq_rx", AdifType.Number, to_mhz),
        "mode": AdifField("mode", AdifType.String),
        "begin_date": AdifField("qso_date", AdifType.Date),
        "begin_time": AdifField("time_on", AdifType.Time),
        "end_time": AdifField("time_off", AdifType.Time),
        "rst_received": AdifField("rst_rcvd", AdifType.String),
        "rst_send": AdifField("rst_sent", AdifType.String),
        "src_call": AdifField("station_callsign", AdifType.String),
        "operator_call": AdifField("operator", AdifType.String),
        "__repeater": AdifField("sat_name", AdifType.String),
        "name": AdifField("name_intl", AdifType.String),
        "comment": AdifField("notes_intl", AdifType.String),
        "__prop_mode": AdifField("prop_mode", AdifType.String),
        "rig": AdifField("my_rig", AdifType.String),
        "antenna": AdifField("my_antenna", AdifType.String),
        "dst_locator": AdifField("gridsquare", AdifType.String),
        "manager": AdifField("qsl_via", AdifType.String),
        "contest_ex_tx": AdifField("stx_string", AdifType.String),
        "contest_ex_rx": AdifField("srx_string", AdifType.String),
    }

    adif_darc_qsl = {
        "__qsl_rcvd": AdifField("qsl_rcvd", AdifType.Enumeration),
        "qslcomment": AdifField("notes_intl", AdifType.String),
    }

    def __init__(self):
        self.args = {}
        self.config_values = read_config()
        self.session_id = os.environ.get("7LOG_SESSION_ID")
        if not self.session_id:
            session_id = self.config_values.get("session_id")
        if not self.session_id:
            # username for windows
            self.session_id = os.environ.get("USER") or os.environ.get("USERNAME")
        if not self.session_id:
            print(
                "No session id. Please set session_id in config or 7LOG_SESSION_ID environment variable",
                file=sys.stderr,
            )
            raise ConfigException()
        qrz_login = self.config_values.get("qrzlogin")
        qrz_pw = self.config_values.get("qrzpassword")
        self.qrz_access = None
        if qrz_pw and qrz_login:
            self.qrz_access = QrzAccess(qrz_login, qrz_pw)
        self.db = LogDbAccess(self.config_values["database"], session_id)
        self._country_db = None

    def country_db(self):
        if self._country_db is None:
            self._country_db = DXCCData(country_data_files())
        return self._country_db

    def _parse_cmd_line(self):
        ap = argparse.ArgumentParser(
            description="7log Version: %s" % _version,
            epilog="Arguments with [D] are qso data attributes."
            " Please store the last qso with -s",
        )
        ap.add_argument(
            "--antenna", help="[D] Antenna overwrites from --rig rigdesc.json"
        )
        ap.add_argument(
            "-c", "--call", dest="call", help="[D] Callsign of other station"
        )
        ap.add_argument(
            "--cc", action="store_true", help="Preset call from dxcluster info"
        )
        ap.add_argument("--date", dest="date", help="[D] Date of qso in yyyy-mm-dd")
        ap.add_argument("-i", "--info", dest="info", help="[D] Additional info for qso")
        ap.add_argument(
            "--finished",
            nargs="?",
            dest="finished",
            const="now",
            help="[D] Finish time of qso in hh:mm or NOW (default)",
        )
        ap.add_argument(
            "--locator", dest="locator", help="[D] Locator of other station"
        )
        ap.add_argument("--manager", dest="manager", help="[D] QSL manager")
        ap.add_argument(
            "-m", "--mode", dest="mode", help="[D] Modulation or mode used for qso"
        )
        ap.add_argument("-n", "--name", dest="name", help="[D] Name of operator")
        ap.add_argument(
            "-r", "--repeater", dest="repeater", help="[D] Repeater used for qso"
        )
        ap.add_argument(
            "--report",
            dest="report_send",
            help="[D] Given report RST, Defaults to 59(9)",
        )
        ap.add_argument("--rst", dest="report_rcvd", help="[D] Receive report")
        ap.add_argument("--time", dest="time", help="[D] Time of qso [UTC] in hh:mm")
        # Store and update data
        ap.add_argument(
            "--cancel",
            action="store_true",
            help="Removes last uncommited qso from storage",
        )
        ap.add_argument(
            "--correct",
            dest="correct",
            help="Correct callsign of other station. Overwrites current do not commit previous qso.",
        )
        ap.add_argument(
            "--delete", dest="delete", type=int, help="Delete record with given number."
        )
        ap.add_argument(
            "--repeatersearch", help="Search nearest repeater by locator or frequency"
        )
        ap.add_argument(
            "-s",
            "--store",
            action="store_true",
            help="Stores current record / commits to db without new callsign",
        )
        ap.add_argument(
            "--update",
            type=int,
            dest="update_record",
            help="Update given qso number with following arguments",
        )
        # show data
        ap.add_argument(
            "--configuration",
            action="store_true",
            help="list all configuration options",
        )
        ap.add_argument(
            "--last",
            nargs="?",
            const=10,
            type=int,
            help="List given number of last qsos.",
        )
        ap.add_argument(
            "--list",
            dest="qsolist",
            metavar=("<start date>", "<end date>"),
            nargs=2,
            help="list qsos.Argument are <from date > <to date> to_date may be an empty string.",
        )
        ap.add_argument("-/", "--search", help="Search database for string")
        ap.add_argument(
            "--printqso",
            nargs="+",
            metavar="<qso numbers>",
            help="Print complete qso data by qso number",
        )
        # export data
        ap.add_argument(
            "--adif",
            nargs=3,
            metavar=("<start date>", "<end date>", "<filename>"),
            help="Export qsos.Argument are <from date > <to date> <file> to_date may be an empty string. If file is empty prints to stdout.",
        )
        ap.add_argument(
            "--qslexport",
            nargs=3,
            metavar=("<src_call>", "<src_locator (maybe empty)>", "<filename or path>"),
            help="Export_qslcards",
        )
        ap.add_argument(
            "--qslsend",
            type=int,
            metavar="<Export job id>",
            help="Mark qslexport job as transmitted and set qslout for the qsos",
        )
        ap.add_argument(
            "--qsllistexport", action="store_true", help="Show all export jobs"
        ),
        ap.add_argument(
            "--qslrecord",
            choices=("pse", "tnx"),
            help="Sets qslin for specified qso by call, date parameters",
        ),
        ap.add_argument(
            "--qslcomment", dest="qslcomment", help="[D] Comment on QSO card"
        )
        ap.add_argument(
            "--cloudlog",
            nargs=3,
            metavar=(
                "<config: Use default for standard>",
                "<start date>",
                "<end date>",
            ),
            help="Send qsos.Argument are <from date > <to date> to_date may be an empty string. If file is empty prints to stdout.",
        )
        ap.add_argument(
            "--lotw",
            nargs=3,
            metavar=("<start date>", "<end date>", "<filename>"),
            help="LOTW compatible adif.Arguments are <from date > <to date> <file> to_date may be an empty string.",
        )
        ap.add_argument(
            "--adifimport",
            type=argparse.FileType("r"),
            help="Imports adif file into database",
        )
        ap.add_argument(
            "--dbimport",
            nargs=2,
            metavar=("<dbfile>", "start date"),
            help="Imports qsos from 2nd 7log db. Args: dbfile, startdate",
        )
        ap.add_argument("--contestfilter", help="Only export contest to adif/lotw")
        ap.add_argument(
            "--edi",
            nargs=3,
            metavar=("<contestname>", "<edi template>", "<file prefix>"),
            help="Exports contest with one file per band for vuhf contests. <contestname> <edi template> <file prefix>",
        )
        ap.add_argument(
            "--edinord",
            nargs=3,
            metavar=("<contestname>", "<edi template>", "<file prefix>"),
            help="Exports contest with one file per band for nordcontest. <contestname> <edi template> <file prefix>",
        )
        ap.add_argument(
            "--queryall",
            action="store_true",
            help="query for all calls, (adif lotw list)",
        )
        ap.add_argument(
            "--rig",
            help="Rigname direct or from rigdesc.json, if not given defaults to defaultrig from config",
        )
        ap.add_argument(
            "--rotator", nargs=1, help="Set rotator to given locator or azimuth"
        )
        ap.add_argument("--qrz", nargs=1, help="Query call sign in qrz.com")
        ap.add_argument(
            "--area",
            nargs=1,
            metavar="<call/prefix>",
            help="Shows Country and area information for call",
        )
        ap.add_argument("--map", nargs=1, help="Show Openstreetmap location")
        ap.add_argument(
            "--set",
            dest="setting",
            nargs=2,
            metavar=("<name>", "<value>"),
            help="set setting name=value. valid names: callsign, operator, locator, lastqsos, qrzlogin, qrzpassword, rigctlhost, defaultrig or database,",
        )
        ap.add_argument("--riglist", action="store_true", help="list configured rigs")
        # Marker functions
        ap.add_argument(
            "--mark",
            nargs="+",
            help="Set marker for specified call on current frequency. At least call is necessary",
        )
        ap.add_argument("--delmarks", action="store_true", help="Delete all markers")
        ap.add_argument(
            "--listmarks",
            action="store_true",
            help="List all markers sort by frequency",
        )
        ap.add_argument("--findmarks", nargs=1, help="search for call in makers")
        ap.add_argument(
            "--stations",
            action="store_true",
            help="Print stations on current frequency from markers",
        )
        # contest functions
        ap.add_argument(
            "--contestnew",
            nargs=2,
            metavar=("<template name>", "<contest name>"),
            help="creates new contest from template. Args: templatename, newname",
        )
        ap.add_argument(
            "--contestupdate",
            nargs=2,
            metavar=("<template name>", "<contest name>"),
            help="updates contest from template. Args: templatename, contest to update",
        )
        ap.add_argument(
            "--contestlist",
            action="store_true",
            help="list templates and known contests",
        )
        ap.add_argument(
            "--contestinfo",
            metavar="<contest name>",
            help="shows contest info / number per band",
        )
        ap.add_argument(
            "--contestset",
            metavar="<contest name>",
            help="sets session into contest mode by contestname",
        )
        ap.add_argument(
            "--contest",
            metavar="<contest name>",
            help="use specified contest for logging (overwrites contestset)",
        )
        ap.add_argument(
            "--grabcluster",
            action="store_true",
            help="grep dxluster infos via telnet formdxclusterhost dxclusterport",
        )
        ap.add_argument(
            "--fetchareadata", action="store_true", help="Loads area data from cqrlog"
        )
        ap.add_argument(
            "--who",
            const="-1",
            nargs="?",
            metavar="<freqency like 14.125Mhz>",
            help="Show station on current frequency or given from dx cluster info",
        )
        ap.add_argument(
            "--gotocall",
            nargs=2,
            metavar=("<call>", "<Band use . for current band. A matches all>"),
            help="Goto frequency of given station",
        )
        ap.add_argument(
            "--dxcluster",
            metavar="<band>",
            help="shows cluster spots grabbed for band (like 20m)",
        )
        ap.add_argument("--spotqso", help="Send DX cluster spot for current qso")

        ap.add_argument(
            "cmdargs",
            nargs="*",
            help="Args to guess  per regex or a n:<name>, c:<call>, "
            "r:<repeater>, s:<report send> /<contest_exchange b:<band>. "
            "Unknown args are added to the comment field. "
            "If name is not given, the name is requested from qrz.com. "
            "Regexp for call, frequency in [x]Hz, rst_send, mode, date, "
            "time, 'qslin', 'qslout', 'qslreq'. Date/time format is yyyy-mm-dd hh:mm",
        )

        ns = ap.parse_args()
        return ns

    def src_call_filter(self):
        src_call = None
        if not self.args.queryall:
            src_call = self.config_values["callsign"]
        return src_call

    def handle_cmds(self):
        """ """
        is_cmd = True
        if self.args.configuration:
            self.show_config()
        elif self.args.setting:
            self.set_configuration(self.args.setting)
        elif self.args.cancel:
            self.cancel_record()
        elif self.args.store:
            self.commit_record()
        elif self.args.qsolist:
            self.list_qsos(self.args.qsolist, self.src_call_filter())
        elif self.args.adif:
            self.adif_export(
                self.args.adif, self.args.contestfilter, False, self.src_call_filter()
            )
        elif self.args.lotw:
            self.adif_export(
                self.args.lotw, self.args.contestfilter, True, self.src_call_filter()
            )
        elif self.args.qslexport:
            self.qsl_export(self.args.qslexport)
        elif self.args.qslsend is not None:
            self.qsl_export_done(self.args.qslsend)
        elif self.args.qsllistexport:
            self.show_export_jobs()
        elif self.args.cloudlog:
            self.cloudlog_export(
                self.args.cloudlog, self.args.contestfilter, self.src_call_filter()
            )
        elif self.args.edi:
            edi_export(
                self.db,
                self.args.edi[0],
                self.args.edi[1],
                self.args.edi[2],
                EDITYPE.EDI_VUHF,
            )
        elif self.args.edinord:
            edi_export(
                self.db,
                self.args.edinord[0],
                self.args.edinord[1],
                self.args.edinord[2],
                EDITYPE.EDI_NORD,
            )
        elif self.args.delete:
            self.delete_qso(self.args.delete)
        elif self.args.correct:
            self.correct_call(self.args.correct)
        elif self.args.qrz:
            self.show_qrz(self.args.qrz)
        elif self.args.area:
            self.show_area(self.args.area)
        elif self.args.search:
            self.search_db(self.args.search)
        elif self.args.printqso:
            self.print_qsos(self.args.printqso)
        elif self.args.rotator:
            self.set_rotator(self.args.rotator[0], self.args.rig)
        elif self.args.map:
            self.show_map(self.args.map[0])
        elif self.args.mark:
            self.add_mark(self.args.mark, self.args.rig)
        elif self.args.delmarks:
            self.del_marks()
        elif self.args.listmarks:
            self.list_marks()
        elif self.args.findmarks:
            self.find_marks(self.args.findmarks[0])
        elif self.args.stations:
            self.print_stations(self.args.rig)
        elif self.args.last:
            self.print_last_qsos(self.args.last)
        elif self.args.contestlist:
            self.print_contests()
        elif self.args.contestnew:
            self.create_or_update_contest(
                self.args.contestnew[0], self.args.contestnew[1], True
            )
        elif self.args.contestupdate:
            self.create_or_update_contest(
                self.args.contestupdate[0], self.args.contestupdate[1], False
            )
        elif self.args.contestinfo:
            self.show_contest_info(self.args.contestinfo)
        elif self.args.contestset is not None:
            self.set_contest_mode(self.args.contestset)
        elif self.args.riglist:
            self.print_rigs()
        elif self.args.repeatersearch:
            self.print_repeater(self.args.repeatersearch)
        elif self.args.adifimport:
            self.import_adif(self.args.adifimport)
        elif self.args.dbimport:
            self.import_from_db(self.args.dbimport[0], self.args.dbimport[1])
        elif self.args.grabcluster:
            self.grab_cluster()
        elif self.args.fetchareadata:
            self.fetch_area_data()
        elif self.args.dxcluster:
            self.show_cluster(self.args.dxcluster)
        elif self.args.who:
            self.who(self.args.who)
        elif self.args.gotocall:
            self.gotocall(self.args.gotocall)
        elif self.args.spotqso:
            self.spot_qso(self.args.spotqso)
        elif self.args.qslrecord:
            self.qsl_record(self.args.qslrecord)
        else:
            is_cmd = False
        return is_cmd

    def show_cluster(self, band):
        """
        shows spot on given band
        """
        spots = self.db.get_dx_cluster_spots(band)
        for sp in spots:
            spot = dict(sp)
            spot["freq"] = spot["frequency"] / 1000.0
            print("{freq:<10} {heardcall:<10} {comment:<35} {location:}".format(**spot))

    def _cluster_config(self):
        host = self.config_values.get("dxclusterhost")
        port = self.config_values.get("dxclusterport")
        loginname = self.config_values.get("dxclustercall")
        if host and port and port.isdigit() and loginname:
            return host, int(port), loginname
        else:
            print(
                "Invalid cluster configuration host:{}, port:{}, call:{}".format(
                    host, port, loginname
                )
            )
            return None, None, None

    def spot_qso(self, message):
        """
        send spot to dx_cluster
        this stops the grabbing session-
        only one connection per call is allowed
        """
        currqso = self.db.current_qso()
        if currqso is None:
            currqso = self.db.last_qso()
        if currqso is not None:
            qsodict = dict(currqso)
            call = qsodict.get("dst_call")
            freq = qsodict.get("freq")
            if call and freq:
                host, port, loginname = self._cluster_config()
                if host is not None:
                    dx = DxCluster(host, int(port), loginname)
                    spot = dx.spot_qso(freq, call, message)
                    print("SPOT: {}".format(spot))

    def grab_cluster(self):
        host, port, loginname = self._cluster_config()
        if host is not None:
            dx = DxCluster(host, int(port), loginname)
            try:
                dx.connect()
                dx.loginDX()
                dx.startxmlrpcserver()
                for dxspot in dx.read_dx():
                    delete_from = datetime.datetime.utcnow() - datetime.timedelta(
                        hours=1
                    )
                    dxspot["band"] = frequency_to_band(dxspot["frequency"])
                    self.db.insert_dxcluster_spot(dxspot, delete_from)
            except KeyboardInterrupt:

                pass
            finally:
                dx.shutdown_xml()
                dx.disconnect()

    def fetch_area_data(self):
        config_dir = get_conf_directory()
        au = AreaUpdater()
        au.updateAreaDb(config_dir)

    def show_cluster_for_frequency(self, f):
        cDB = self.country_db()
        spots = self.db.get_dx_cluster_calls(f)
        spotted_calls = set()
        for spot in spots:
            heardcall = spot["heardcall"]
            if heardcall:
                heardcall = heardcall.upper().strip()
                datestr = spot["spotdate"]
                if datestr:
                    date = datetime.datetime.fromisoformat(datestr)
                else:
                    date = datetime.datetime.now()
                print(
                    "'{}' spotted at {:02d}:{:02d}Z by {}".format(
                        heardcall, date.hour, date.minute, spot["srccall"]
                    )
                )
                if heardcall not in spotted_calls:
                    spotted_calls.add(heardcall)
                    country_data = cDB.call_data(heardcall)
                    if country_data:
                        az = self.bearing_from_lat_lon(
                            country_data.lat, country_data.lon
                        )
                        print("{0} AZ:{1:0.1f}°".format(country_data, az))
                    self.show_last_qsos(heardcall)

    def _get_rig_ctl(self, msg_cmd):
        rig_name = self.args.rig
        rigctld = None
        if rig_name is None:
            rig_name = self.config_values.get("defaultrig")
        if rig_name is not None:
            rig_info = read_rig(rig_name)
            if rig_info is not None:
                rigctld = rig_info.rigctrl()
        if not rigctld:
            rigctld = self.config_values.get("rigctlhost")
        if not rigctld:
            print(f"{msg_cmd} needs configured rigctld")
        return rigctld, rig_info

    def who(self, who):
        f = None
        if who != "-1":
            f = self.frequ_in_hz(who)
        if not f:
            rig_name = self.args.rig
            rigctld = None
            if rig_name is None:
                rig_name = self.config_values.get("defaultrig")
            if rig_name is not None:
                rig_info = read_rig(rig_name)
                if rig_info is not None:
                    rigctld = rig_info.rigctrl()
            if not rigctld:
                print("--who needs configured rigctld")
                return
            rigctld, rig_info = self._get_rig_ctl("--who")
            if rigctld is None:
                return
            with HamLibConnection(rigctld) as rig:
                f = rig.read_frequency()
                if f and rig_info is not None:
                    f += rig_info.frequency_offset()
        if f:
            self.show_cluster_for_frequency(f)

    def gotocall(self, args):
        """
        :param args:list of value[0] ==call
          args[1] = band name, . = for current, * for all
        """
        f = None
        rigctld, rig_info = self._get_rig_ctl("--gotocall)")
        if rigctld is None:
            return
        with HamLibConnection(rigctld) as rig:
            f = rig.read_frequency()
            if f and rig_info is not None:
                f += rig_info.frequency_offset()
            bandfilter = args[1]
            if bandfilter.upper() == "A":
                bandfilter = None
            elif bandfilter == ".":
                if f:
                    bandfilter = frequency_to_band(f)
                    if bandfilter == "UNKOWN":
                        print("Not a valid band using any", file=sys.stderr)
                        bandfilter = None
            dxcluster_records = list(
                self.db.get_dx_cluster_call(args[0].upper(), bandfilter)
            )
            if dxcluster_records:
                last_rec = dict(dxcluster_records[0])
                station_qrg = last_rec["frequency"]
                if station_qrg:
                    rig.set_frequency(station_qrg)
                else:
                    print("No frequency in cluster record", file=sys.stderr)
            else:
                print("Call not found", file=sys.stderr)

    def import_from_db(self, dbfile, startdate):
        """
        Import qso fromexternal db
        """
        dbfile = os.path.abspath(dbfile)
        if os.path.isfile(dbfile):
            import_db = LogDbAccess(dbfile, "--")
            if not startdate:
                startdate = "1900-01-01"
            qsos = import_db.get_qsos(startdate, "now", None)
            no_imported = 0
            for qso in qsos:
                qsodict = dict(qso)
                rowid = qsodict.pop("rowid", None)
                band = qsodict.get("band")
                if not band:
                    print("QSO {} without band ignored".format(rowid))
                else:
                    if self.db.addIfNotExists(qsodict):
                        no_imported += 1
            print("{} QSOs imported".format(no_imported))
        else:
            print(f"File {dbfile} does not exist.")

    def import_adif(self, fileh):
        """
        import adif records into db if
        qso doesnt exist in db.
        does not overwrite qsos that match in date, time,
        src, dst-call, and band.
        """
        try:
            records = Adif.readadif(fileh)
            if records:
                for r in records:
                    dbRec = dict()
                    for dbKey, adifDesc in self._adif_db_map.items():
                        val = r.get(adifDesc.name)
                        if val is not None:
                            if adifDesc.fieldtype == AdifType.Date:
                                val = val[0:4] + "-" + val[4:6] + "-" + val[6:8]
                            elif adifDesc.fieldtype == AdifType.Time:
                                val = val[0:2] + ":" + val[2:4]
                            elif adifDesc.fieldtype == AdifType.Number:
                                val = float(val)
                            if adifDesc.calc_func is not None:
                                val = adifDesc.calc_func(val, inv=True)
                            if dbKey in ["src_call", "dst_call"]:
                                val = val.upper()
                            elif dbKey in ["band"]:
                                val = val.lower()
                            if not dbKey.startswith("__"):
                                dbRec[dbKey] = val
                            elif dbKey == "__repeater":
                                v = r.get("sat_name")
                                if v:
                                    dbRec["repeater"] = v
                    if "src_call" not in dbRec:
                        dbRec["src_call"] = self.config_values["callsign"].upper()
                    self.db.addIfNotExists(dbRec)
        finally:
            fileh.close()

    def set_contest_mode(self, contestname):
        c_info = self.db.contest_info(contestname)
        if c_info is not None or contestname == "":
            self.db.set_contest_mode(contestname)
        else:
            print("Contest {} does not exists".format(contestname))

    def show_contest_info(self, contestname):
        """
        shows info for contest
        """
        c_info = self.db.contest_info(contestname)
        if c_info is not None:
            print("Number scheme: {}".format(c_info.number_scheme))
            print("Band  Current number")
            for band, number in self.db.contest_numbers(contestname):
                print("{band:<5}  {number:<6}".format(band=band, number=number))

        else:
            print("Contest {} does not exists".format(contestname))

    def print_contests(self):
        """
        print known contest templates and create contest instances
        """
        contest_templates = read_contest_templates()
        print("Templates")
        for contest_name in contest_templates.keys():
            print("  {}".format(contest_name))
        print("Contests")
        for c_row in self.db.known_contests():
            print("  {}".format(c_row["contest_name"]))
        curr = self.db.current_contest()
        if curr:
            print("Active contest: {}".format(curr))

    def print_rigs(self):
        """
        print configured rignames
        """
        rigs = rig_names()
        print("Rigs:")
        for r in rigs:
            print("   {}".format(r))

    def create_or_update_contest(self, template, contestname, create):
        """
        creates a contest from template with contestname
        """
        contest_templates = read_contest_templates()
        if not contest_templates:
            print("No contest templates found (./config/7log/contesttemplate.json")
        else:
            c_template = contest_templates.get(template)
            if c_template is None:
                print("Unknown contest template {}".format(template))
            else:
                # as for varibales
                number_scheme = c_template.number_scheme
                exchange_data = c_template.exchange_data
                for var in c_template.scheme_variables():
                    if var != ("$(NUM)"):
                        value = input("Variable: {}:".format(var))
                        number_scheme = number_scheme.replace(var, value)
                        exchange_data = exchange_data.replace(var, value)
                print("number scheme: {}".format(number_scheme))
                print("exchange_data:  {}".format(exchange_data))
                if create:
                    self.db.create_contest(
                        contestname,
                        c_template.number_per_band,
                        number_scheme,
                        exchange_data,
                        c_template.required_attrs,
                        c_template.unique_station_attrs,
                    )
                else:
                    self.db.update_contest(
                        contestname,
                        c_template.number_per_band,
                        number_scheme,
                        c_template.required_attrs,
                        c_template.unique_station_attrs,
                    )

    def add_mark(self, cmd_args, rig_name):
        matched, _unmatched = guess_args(cmd_args)
        freq_str = matched.get("freq")
        mark_attrs = []
        mark_attrs.extend(cmd_args)
        if mark_attrs:
            config_values = read_config(False)
            rig_info = None
            if rig_name is None:
                rig_name = config_values.get("defaultrig")
            if rig_name is not None:
                rig_info = read_rig(rig_name)
            if freq_str is None:
                vals = self._add_defaults(matched, rig_info)
                frequency = vals.get("freq")
            else:
                frequency = self.frequ_in_hz(freq_str)
            if frequency is not None:
                band = frequency_to_band(frequency)
                qtf = None
                t_def = time_defaults()
                time_stamp = t_def["time"]
                if rig_info is not None:
                    rot_host = rig_info.rotctrl()
                    if rot_host:
                        with HamLibConnection(rot_host) as rot:
                            qtf, _el = rot.rot_get_pos()
                self.db.update_or_insert_marker(
                    " ".join(mark_attrs), band.upper(), frequency, qtf, time_stamp
                )
        else:
            print("Mark needs parameter as description")

    def del_marks(self):
        n = self.db.delete_markers()
        if n > 0:
            v = ""
            while v.upper() not in ["Y", "N"]:
                v = input("{} markers will be deleted, Really delete? (y/n) ".format(n))
            if v.upper() == "Y":
                self.db.conn.commit()
            else:
                self.db.conn.rollback()
        else:
            self.db.conn.rollback()

    def print_marker(self, mark_record):
        values = {
            "frequency": mark_record["frequency"] / 1.0e6,
            "call": mark_record["call"],
            "qtf": mark_record["qtf"] if mark_record["qtf"] is not None else 0.0,
            "ts": mark_record["timestamp"],
        }
        print("M: {frequency:5.3F} MHz {call:<6} {qtf:3.1F}° {ts}".format(**values))

    def list_marks(self):
        """
        List all marked stations
        """
        for m in self.db.list_markers():
            self.print_marker(m)

    def find_marks(self, call):
        for m in self.db.find_marker_by_call(call):
            self.print_marker(m)

    def print_stations(self, rig_name):
        """
        Plotting stations on current frequency
        """
        rigctld = None
        if rig_name is None:
            rig_name = self.config_values.get("defaultrig")
        if rig_name is not None:
            rig_info = read_rig(rig_name)
            if rig_info is not None:
                rigctld = rig_info.rigctrl()
        if rigctld is None:
            print("--stations needs configured rigctld")
            return
        print("Show marked stations. Press CTRL c to stop")
        try:
            last_f = 0.0
            with HamLibConnection(rigctld) as rig:
                while True:
                    f = rig.read_frequency()
                    if f is not None:
                        if abs(f - last_f) > 300:
                            last_f = f
                            stations = self.db.find_marker_by_frequency(f)
                            for s in stations:
                                self.print_marker(s)
                            self.show_cluster_for_frequency(f)
                    time.sleep(0.5)
        except KeyboardInterrupt:
            pass

    def set_rotator(self, destination, rig):
        """
        Sets rotator to given locator or degrees
        """
        config_values = read_config()
        az = None
        if destination.isdigit():
            # assumen degress
            az = float(destination)
        elif destination:
            own_loc = config_values.get("locator")
            dstloc = toLoc(destination)
            if own_loc and dstloc:
                az = bearing_degrees(toLoc(own_loc), dstloc)
        if az is not None:
            if rig is None:
                rig = config_values.get("defaultrig")
            if rig is None:
                print("--rotator needs configured or defaultrig or --rig")
            else:
                rig_config = read_rig(rig)
                rothost = rig_config.rotctrl()
                if not rothost:
                    print("--rotator needs configured rotator for rig %s" % rig)
                else:
                    with HamLibConnection(rothost) as rot:
                        ok = rot.rot_set_pos(az, 0.0)
                        if ok:
                            print("Rotator set to %.0f" % az)
                        else:
                            print("Rotator set failed (invalid az?)")

    def search_db(self, s):
        """ """
        show_locator = bool(self.db.current_contest())
        self.show_qso(self._header, show_locator=show_locator)
        for row in self.db.search_qsos(s):
            self.show_qso(row, show_locator=show_locator)

    def print_last_qsos(self, max_number):
        """
        print number of last qsos
        """
        show_locator = bool(self.db.current_contest())
        self.show_qso(self._header, show_locator=show_locator)
        last_qsos = list(self.db.last_qsos(max_number))
        last_qsos.reverse()
        for row in last_qsos:
            self.show_qso(row, show_locator=show_locator)

    def print_qsos(self, qso_numbers):
        """
        :param qso_numbers: list of qso_numbers as string
        """
        qso_num_int = []
        neg_values = []
        for qso in qso_numbers:
            try:
                qso_int = int(qso)
            except:
                continue
            if qso_int < 0:
                neg_values.append(qso_int)
            else:
                qso_num_int.append(qso_int)
        if neg_values:
            l_qso = list(self.db.last_qsos(1))
            if l_qso:
                qso_dict = dict(l_qso[-1])
                last_row = qso_dict.get("rowid")
                if last_row is not None:
                    for n_val in neg_values:
                        qso_num = last_row + 1 + n_val
                        qso_num_int.append(qso_num)
        qso_num_int.sort()
        prev_qso = False
        for qso in qso_num_int:
            if prev_qso:
                print("")
            prev_qso = self.print_qso(qso)

    def print_qso(self, qso_number):
        """
        complete data of a single qso
        """
        qso_rec = self.db.get_qso_by_number(qso_number)
        if qso_rec is None:
            print(f"{qso_number} does not exist, number ignored", file=sys.stderr)
        else:
            self.print_str_attr(qso_rec, "My Call", "src_call")
            self.print_str_attr(qso_rec, "Operator", "operator_call")
            self.print_str_attr(qso_rec, "Station", "dst_call")
            self.print_str_attr(qso_rec, "Date", "begin_date")
            self.print_str_attr(qso_rec, "Time", "begin_time")
            self.print_str_attr(qso_rec, "RST send", "rst_send")
            self.print_str_attr(qso_rec, "Contest exch.", "contest_ex_tx")
            self.print_str_attr(qso_rec, "RST rcvd", "rst_received")
            self.print_str_attr(qso_rec, "Contest exch.", "contest_ex_rx")
            self.print_str_attr(qso_rec, "Repeater/SAT", "repeater")
            self.print_str_attr(qso_rec, "Band", "band")
            self.print_str_freq(qso_rec, "Frequency", "freq")
            self.print_str_attr(qso_rec, "Band(RX)", "band_rx")
            self.print_str_freq(qso_rec, "Frequency(RX)", "freq_rx")
            self.print_str_attr(qso_rec, "Op name", "name")
            self.print_str_attr(qso_rec, "Info", "comment")
            self.print_locator(qso_rec)
            self.print_str_attr(qso_rec, "Continent", "continent")
            self.print_str_attr(qso_rec, "CQ zone", "dst_cqzone")
            self.print_str_attr(qso_rec, "ITU zone", "dst_ituzone")
            self.print_str_attr(qso_rec, "Contest name", "contest_name")
            self.print_str_attr(qso_rec, "QSL manager", "manager")
            self.print_str_attr(qso_rec, "QSL request", "qslreq")
            self.print_str_attr(qso_rec, "QSL send", "qslout", True)
            self.print_str_attr(qso_rec, "QSL received", "qslin", True)
            self.print_str_attr(qso_rec, "QSL Comment", "qslcomment", True)
            self.print_str_attr(qso_rec, "My RIG", "rig")
            self.print_str_attr(qso_rec, "My antenna", "antenna")
            return True

    def print_str_freq(self, qso_rec, label, attr):
        value = qso_rec[attr]
        if value and isinstance(value, (float, int)):
            f = value / 1000.0
            print(f"{label:<15}: {f}Mhz")

    def print_str_attr(self, qso_rec, label, attr, force_show=False, end="\n"):
        value = qso_rec[attr]
        if value or force_show:
            if value is None:
                value = "-"
            print(f"{label:<15}: {value}", end=end)

    def print_locator(self, qso_rec):
        my_loc = qso_rec["src_locator"]
        dst_loc = qso_rec["dst_locator"]
        has_distance = my_loc and dst_loc
        end = " / " if has_distance else "\n"
        self.print_str_attr(qso_rec, "My Locator", "src_locator")
        self.print_str_attr(qso_rec, "Station Locator", "dst_locator", end=end)
        if has_distance:
            self.show_distance(my_loc, dst_loc)

    def show_qrz(self, qrz_data):
        if self.qrz_access is None:
            print("Needs username and password for qrz.com (--set ...)")
        else:
            if qrz_data:
                call = qrz_data[0]
                c_info, m_date = self.db.lookup_qrz_cache(call.upper())
                if c_info is not None and m_date is not None:
                    # re request after one year
                    if m_date < (time.time() - 3600 * 24 * 365):
                        c_info = None
                if c_info is None:
                    c_info = {}
                    try:
                        c_info = self.qrz_access.call_info(call)
                    except QRZException as e:
                        print("Error from qrz.com: {}".format(str(e)))
                    if c_info:
                        self.db.update_qrz_cache(call.upper(), c_info)
                if c_info:
                    print("qrz.com information:")
                    for k, v in c_info.items():
                        print("{:>10}: {}".format(k, v))

    def bearing_from_lat_lon(self, lat, lon):
        az = 0.0
        if lat is not None and lon is not None:
            own_loc = self.config_values.get("locator")
            dstloc = Point(lon, lat)
            if own_loc and dstloc:
                az = bearing_degrees(toLoc(own_loc), dstloc)
        return az

    def show_area(self, area_data):
        if area_data:
            cDB = self.country_db()
            for call in area_data:
                country_data = cDB.call_data(call)
                if country_data is not None:
                    az = self.bearing_from_lat_lon(country_data.lat, country_data.lon)
                    print("{0} AZ:{1:0.1f}°".format(country_data, az))
                else:
                    print("Unknown country")

    def show_config(self):
        for k, v in self.config_values.items():
            if k in ["qrzpassword", "cloudlogkey"]:
                v = "*******"
            print("%s = %s" % (k, v))

    def set_configuration(self, kv_pair):
        data = read_config(True)
        if kv_pair[0] in data:
            if kv_pair[0] in ["lastqsos"] and kv_pair[1].isdigit():
                v = int(kv_pair[1])
            else:
                v = kv_pair[1]
            set_config_value(kv_pair[0], v)
        else:
            print("Unknown setting %s" % kv_pair[0])
        self.config_values = read_config()

    def cancel_record(self):
        if self.db.is_uncommited():
            self.db.cancel_session()
        else:
            print("Can't cancel qso. No uncommitted qso.")

    def check_complete(self):
        ok = True
        missing = self.db.can_commit()
        if missing is not None:
            if missing:
                ok = False
                missing_values = ", ".join(missing)
                print("Can't complete qso: Please insert: {}".format(missing_values))
        return ok

    def commit_record(self):
        if self.db.is_uncommited():
            if self.check_complete():
                self.db.commit_session()
        else:
            print("Can't store qso. No uncommitted qso.", file=sys.stderr)

    def adif_export(self, para, contest_filter, lotw_mode, src_call):
        config_values = read_config()
        from_date = para[0]
        to_date = para[1]
        fname = para[2]
        # Adif, AdifField, AdifType
        adif = Adif("7log", config_values["callsign"], self._adif_db_map, lotw_mode)
        adif.open_for_write(fname)
        for row in self.db.get_qsos(from_date, to_date, src_call):
            if contest_filter and row["contest_name"] != contest_filter:
                continue
            adif.append_record(row)
        adif.close()

    def qsl_export(self, args):
        """
        Export qsl cards by call and locator
        """
        src_call = args[0].upper()
        src_locator = args[1].upper()
        dst_filename = args[2]
        qso_counter = 0
        if dst_filename in ("", "."):
            dst_filename = os.getcwd()
        p = pathlib.Path(dst_filename)
        timestamp = datetime.datetime.now(datetime.timezone.utc).isoformat(
            timespec="seconds"
        )
        timestamp_file = timestamp.replace(":", "-")
        export_id = self.db.start_qsl_exportjobs(timestamp)
        if p.is_dir():
            f_name = f"qslexport_{export_id}_{timestamp_file}.adi"
            dst_path = p.joinpath(f_name)
        else:
            dst_path = p
        try:
            adif_map = dict()
            adif_map.update(self._adif_db_map)
            adif_map.update(self.adif_darc_qsl)
            adif = Adif("7log", src_call, adif_map, False)
            adif.open_for_write(str(dst_path))
            exported_rows = []
            for row in self.db.request_qsl_cards(src_call, src_locator):
                row_dict = dict(row)
                qso_counter += 1
                # add rxtx and antenne to comment
                antenna = row_dict.get("antenna")
                rig = row_dict.get("rig")

                previous_comment = row_dict.get("qslcomment", "")
                comment_list = []
                if rig:
                    comment_list.append(f"Rig: {rig}")
                if antenna:
                    comment_list.append(f"Ant: {antenna}")
                if previous_comment:
                    comment_list.append(previous_comment)
                if comment_list:
                    row_dict["qslcomment"] = "\r\n".join(comment_list)
                adif.append_record(row_dict)
                exported_rows.append(row["rowid"])
                self.show_qso(row_dict, True, False)
            adif.close()
            self.db.finish_qsl_export_job(export_id, exported_rows, qso_counter == 0)
            if qso_counter > 0:
                print(
                    f"QSL data for {qso_counter} QSOs exported as job {export_id} to {dst_path}"
                )
                print("After upload mark qso as exported with:")
                print(f"7log --qslsend {export_id}")
            else:
                dst_path.unlink()
                print("No QSOs found. Nothing exported.")
        except EnvironmentError:
            self.db.finish_qsl_export_job(export_id, {}, True)
            if dst_path.is_file():
                dst_path.unlink()
            raise

    def qsl_export_done(self, row_id):
        rows = self.db.qsl_export_done(row_id)
        print(f"{rows} qsos marked with qslout")

    def show_export_jobs(self):
        for r in self.db.get_export_jobs():
            print(
                f"Nr: {r['rowid']} at {r['exporttime']}, Committed: {r['jobcommitted']}"
            )

    def cloudlog_export(self, para, contest_filter, src_call):
        config_values = read_config()
        conf_name = para[0]
        from_date = para[1]
        to_date = para[2]
        adif = Adif("7log", config_values["callsign"], self._adif_db_map, False)
        adif_lines = []
        for row in self.db.get_qsos(from_date, to_date, src_call):
            if contest_filter and row["contest_name"] != contest_filter:
                continue
            adif_lines.append(adif.append_record(row, to_file=False))
        if adif_lines:
            cl_config = get_cloud_log(conf_name)
            if cl_config is not None:
                url = cl_config["url"]
                api_key = cl_config["key"]
                station_id = cl_config["stationid"]
                if not url or not api_key or not station_id:
                    print(
                        "Invalid cloudlog-configuration missing value. Check config",
                        file=sys.stderr,
                    )
                else:
                    ok = cloudlog_send_adif(url, api_key, station_id, adif_lines)
                    if ok:
                        print(
                            "{} qso transmitted to cloud log instance {}".format(
                                len(adif_lines), conf_name
                            )
                        )
                    else:
                        print(
                            "Error from cloudlog instance {}".format(conf_name),
                            file=sys.stderr,
                        )
            else:
                print(
                    "cloudlog config {} not found or invalid.".format(conf_name),
                    file=sys.stdout,
                )
        else:
            print("No qsos found", file=sys.stderr)

    def list_qsos(self, para, src_call):
        show_locator = bool(self.db.current_contest())
        self.show_qso(self._header, show_locator=show_locator)
        for row in self.db.get_qsos(para[0], para[1], src_call):
            self.show_qso(row, show_locator=show_locator)

    def delete_qso(self, para):
        if para > 0:
            self.db.delete(para)
        else:
            print("Negative numbers are not implemented", file=sys.stderr)

    def correct_call(self, new_call):
        if self.db.is_uncommited():
            self.db.update_qso({"dst_call": new_call.upper()})
        else:
            print("Can't update callsign No uncommitted qso.", file=sys.stderr)

    def _check_time(self, t_str):
        ok = re.match("^[0-9]{2}:[0-9]{2}$", t_str) is not None

        return ok

    def cluster_call(self):
        """
        try to read call from cluster message
        """
        call = None
        rigctld, rig_info = self._get_rig_ctl("--gotocall)")
        if rigctld is None:
            return
        with HamLibConnection(rigctld) as rig:
            f = rig.read_frequency()
            if f and rig_info is not None:
                f += rig_info.frequency_offset()
        if f:
            dxcalls = list(self.db.get_dx_cluster_calls(f))
            if dxcalls:
                last_rec = dxcalls[-1]
                call = last_rec["heardcall"]
        return call

    def ns_to_db_names(self, input_args):
        """
        Namepspace from cmdline to database record name
        """
        db_names = {
            "call": "dst_call",
            "locator": "dst_locator",
            "date": "begin_date",
            "time": "begin_time",
            "finished": "end_time",
            "report_send": "rst_send",
            "report_rcvd": "rst_received",
            "info": "comment",
        }

        output = dict()
        for k, v in input_args.items():
            if k != "finished":
                output[db_names.get(k, k)] = v
            else:
                f_time = v
                if f_time:
                    if f_time.upper() == "NOW":
                        output["end_time"] = time_defaults()["time"]
                    elif self._check_time(f_time):
                        output["end_time"] = f_time
                    else:
                        print("Invalid time for finished '%s' use hh:mm" % f_time)
        return output

    def upper_case(self, values):
        ucase_fields = ["dst_call", "src_locator", "dst_locator"]
        for f in ucase_fields:
            v = values.get(f)
            if v:
                values[f] = v.upper()
        return values

    def _add_times(self, values):
        t_def = time_defaults()
        d = values.get("begin_date")
        if not d:
            values["begin_date"] = t_def["date"]
        d = values.get("begin_time")
        if not d:
            values["begin_time"] = t_def["time"]
        return values

    def frequ_in_hz(self, f_str):
        """
        :param f_str: String with frequency ending with Hz
        :returns frequency in Hz as float or None
        """
        f = None
        if f_str:
            f_str = f_str.upper()
            # handle MGK, String must end with HZ
            if len(f_str) > 3:
                prefix = f_str[-3:-2]
                f_number = f_str[0:-3]
                f = float(f_number)
                if prefix == "M":
                    f = f * 1e6
                elif prefix == "G":
                    f = f * 1e9
                elif prefix == "K":
                    f = f * 1e3
                elif prefix == "T":
                    f = f * 1e12
                elif prefix.isdigit():
                    f = float(f_str[0:-2])
        return f

    def _add_defaults(self, values, rig_info, current_contest=None):
        """
        add default values from config,
        environment variables or if configured
        from hamlib info.
        """
        db_values = dict()
        db_values.update(values)
        db_values["src_call"] = self.config_values["callsign"]
        db_values["operator_call"] = self.config_values["operator"]
        db_values["src_locator"] = self.config_values["locator"]
        # Handle frequency
        f_str = values.get("freq")
        f = None
        band = None
        f_rx = None
        band_rx = None
        if f_str:
            f = self.frequ_in_hz(f_str)
        else:
            # must be in Hz
            f_str = os.environ.get("DG7BBP_LOG_QRG")
            if f_str:
                f = float(f_str)
            if not f_str:
                rigctlhost = None
                if rig_info is not None:
                    rigctlhost = rig_info.rigctrl()
                if rigctlhost is None:
                    rigctlhost = self.config_values.get("rigctlhost")
                if rigctlhost:
                    with HamLibConnection(rigctlhost) as hamlib:
                        mode_tuple = hamlib.get_mode()
                        if mode_tuple:
                            mode = mode_tuple[0]
                            if mode:
                                # filter out data mode from icom
                                if mode.upper().startswith("PKT"):
                                    mode = mode[3:]
                                if mode.upper() in ["USB", "LSB"]:
                                    mode = "SSB"
                                db_values["mode"] = mode
                        f = hamlib.read_frequency()
                        if f and rig_info is not None:
                            f += rig_info.frequency_offset()
        if f:
            db_values["freq"] = f
            band = frequency_to_band(f)
            db_values["band"] = band
        f_rx_str = os.environ.get("DG7BBP_LOG_QRG_RX")
        if f_rx_str:
            f_rx = float(f_rx_str)
        if f_rx:
            db_values["freq_rx"] = f_rx
            band_rx = frequency_to_band(f_rx)
            db_values["band_rx"] = band_rx
        if rig_info is not None:
            rig = rig_info.get_rigdesc()
            db_values["rig"] = rig
            if db_values.get("antenna") is None:
                db_values["antenna"] = rig_info.get_antenna(db_values.get("band"))
        last_qso = self.db.last_or_current()
        self.add_repeater_from_list(db_values)
        # Handle defaults from last qso
        if last_qso:
            if band is None:
                band = last_qso["band"]
                db_values["freq"] = last_qso["freq"]
            if band is not None:
                db_values["band"] = band
            # think about setting band rx from precious
            # qso. I trx/rigconfig supports badn split
            # may take this info from trx
            # better we have a cmd_line args for this
            # band rx is not supported for manual mode,
            # no update from previous qso
            # if band_rx is None:
            #     band_rx = last_qso["band_rx"]
            #     if last_qso["freq_rx"]:
            #         db_values["freq_rx"] = last_qso["freq_rx"]
            # if band_rx is not None:
            #     db_values["band_rx"] = band_rx
            mode = db_values.get("mode")
            if mode is None:
                mode = last_qso["mode"]
                if mode is not None:
                    db_values["mode"] = mode
            repeater = db_values.get("repeater")
            if repeater is None:
                repeater = last_qso["repeater"]
                band_change = last_qso["band"] != band
                frequ_changed = False
                if db_values.get("freq") and mode in ["FM", "DV"]:
                    frequ_changed = abs(db_values["freq"] - last_qso["freq"]) > 1e3
                if repeater is not None and not band_change and not frequ_changed:
                    db_values["repeater"] = repeater
            rig = db_values.get("rig")
            if rig is None:
                rig = last_qso["rig"]
                if rig is not None:
                    db_values["rig"] = rig
            antenna = db_values.get("antenna")
            if antenna is None:
                antenna = last_qso["antenna"]
                if antenna is not None:
                    db_values["antenna"] = antenna
        rst = db_values.get("rst_send")
        if not rst:
            mode = db_values.get("mode")
            if mode in ["CW", "RTTY"]:
                db_values["rst_send"] = "599"
            else:
                db_values["rst_send"] = "59"
        rst = db_values.get("rst_received")
        if not rst:
            mode = db_values.get("mode")
            if mode == "CW":
                db_values["rst_received"] = "599"
            else:
                db_values["rst_received"] = "59"
        db_values = self.upper_case(db_values)
        call = db_values.get("dst_call")
        name_by_cli = db_values.get("name")
        if call and self.qrz_access and not name_by_cli:
            c_info = None
            c_info, m_date = self.db.lookup_qrz_cache(call)
            if c_info is None or m_date < (time.time() - 365 * 24 * 3600):
                try:
                    c_info = self.qrz_access.call_info(call.lower())
                except QRZException as e:
                    print("Error from QRZ.COM: {}".format(str(e)))
            if c_info:
                fname = c_info.get("fname")
                name = c_info.get("name")
                if fname and name:
                    print("QRZ: Name: {} {}".format(fname, name))
                    db_values["name"] = fname
                self.db.update_qrz_cache(call, c_info)
        if call:
            country_data = self.country_db().call_data(call)
            if country_data is not None:
                if not db_values.get("continent"):
                    db_values["continent"] = country_data.continent
                if not db_values.get("dst_cqzone"):
                    db_values["dst_cqzone"] = country_data.cqzone
                if not db_values.get("dst_ituzone"):
                    db_values["dst_ituzone"] = country_data.ituzone
                print(country_data)
        # update contestinfo
        if current_contest is None:
            contestname = self.db.current_contest()
        else:
            contestname = current_contest
        if contestname:
            db_values["contest_name"] = contestname
            rst_send = db_values["rst_send"]
            contest_ex_tx = ""
            if rst_send:
                if len(rst_send) <= 3:
                    # nur 59 oder 599, dann Zahlen anhaengen
                    c_info = self.db.contest_info(contestname)
                    if c_info is not None:
                        number_scheme = c_info.number_scheme.upper()
                        if number_scheme.find("$(NUM)") >= 0:
                            if c_info.number_per_band:
                                band_id = db_values["band"]
                            else:
                                band_id = "ALL"
                            curr_number = self.db.get_contest_number(
                                contestname, band_id
                            )
                            n = number_scheme.replace(
                                "$(NUM)", "{:03d}".format(curr_number + 1)
                            )
                            contest_ex_tx = n
                        else:
                            contest_ex_tx = number_scheme
                        db_values["contest_ex_tx"] = contest_ex_tx
                    else:
                        print("Current contest does not exists??")
        return db_values

    def show_last_qsos(self, call):
        cnt = self.config_values.get("lastqsos", 0)
        first = True
        for row in self.db.get_last_qsos(call, cnt):
            r_dict = {}
            for k, v in dict(row).items():
                if v is None:
                    v = ""
                elif k.startswith("qsl"):
                    if v:
                        v = "x"
                    else:
                        v = ""
                r_dict[k] = v
            if first:
                print("Last QSOs:")
                first = False
            cstr = "{begin_date:<10}|{begin_time:<5}|{dst_call:<8}|{band:<5}|{name},{comment}".format(
                **r_dict
            )
            print(cstr)

    def show_map(self, loc):
        center = None
        if len(loc) == 4:
            center = loc + "ll"
        elif len(loc) > 5:
            center = loc
        if center:
            p = toLoc(center)
            url = "https://openstreetmap.org/?mlat={lat}&mlon={lon}#map=2/{lat:.4}/{lon:.4}".format(
                lat=p.y, lon=p.x
            )
            if sys.platform.startswith("win"):
                os.startfile(url)
            else:
                prgs = ["/usr/bin/chromium-browser", "/usr/bin/firefox"]
                for browser in prgs:
                    if os.path.isfile(browser):

                        subprocess.run(
                            [browser, url],
                            stdout=subprocess.DEVNULL,
                            stderr=subprocess.DEVNULL,
                        )
                        break

    def show_qso(self, row, with_rowid=True, show_locator=False):
        """
        Display QSO line
        """
        r_dict = {
            "dst_locator": "",
            "contest_ex_rx": "",
            "contest_ex_tx": "",
            "rowid": "",
        }
        for k, v in dict(row).items():
            if v is None:
                v = ""
            elif k.startswith("qsl"):
                if v:
                    v = "x"
                else:
                    v = ""
            r_dict[k] = v
        # print(r_dict)
        if with_rowid:
            cstr = "{rowid:<4}|"
        else:
            cstr = ""
        if show_locator:
            cstr += (
                "{begin_date:<10}|{begin_time:<5}|"
                + "{dst_call:<8}|{mode:<4}|{dst_locator:<6}|{band:<5}|"
                + "{rst_send:<4}{contest_ex_tx}|{rst_received:<4}{contest_ex_rx}|{name:<8}|{qslin:<1}|"
                + "{qslout:<1}|{qslreq:<1}|{comment}"
            )
        else:
            cstr += (
                "{begin_date:<10}|{begin_time:<5}|"
                + "{dst_call:<8}|{mode:<4}|{repeater:<6}|{band:<5}|"
                + "{rst_send:<4}|{rst_received:<4}|{name:<8}|{qslin:<1}|"
                + "{qslout:<1}|{qslreq:<1}|{comment}|{manager}"
            )
        print(cstr.format(**r_dict))

    def split_contest_exc(self, rst, mode):
        contest_exchange = None
        splitted = rst.split("/")
        if len(splitted) == 2:
            contest_exchange = splitted[1]
        rst = splitted[0]
        if not contest_exchange:
            if mode in ["CW", "RTTY"]:
                rstlen = 3
            else:
                rstlen = 2
            if len(rst) > rstlen:
                contest_exchange = rst[rstlen:]
        return rst, contest_exchange

    def db_operations(self, db_values, rig_info, contest=None):
        curr_contest = None
        if contest is not None:
            if contest:
                c_info = self.db.contest_info(contest)
                if not c_info:
                    print(f"contest {contest} unkown. No log.")
                    return
                else:
                    curr_contest = contest
            else:
                curr_contest = None
        else:
            curr_contest = self.db.current_contest()
        if curr_contest:
            self.db_operations_contest(curr_contest, db_values, rig_info)
        else:
            self.db_operations_normal(db_values, rig_info)

    def db_operations_contest(self, curr_contest, db_values, rig_info):
        """
        store data in db contest mode
        in contestmode only complete lines are allowed.
        no extra updates are valid
        if call == lastcall band = last_band an update is performend if band
        """
        db_values = self.upper_case(db_values)
        rst_received = db_values.get("rst_received")
        db_values = self._add_defaults(db_values, rig_info, curr_contest)
        if rst_received is None:
            db_values.pop("rst_received", None)
        elif rst_received and not db_values.get("contest_ex_rx"):
            rst_received, ctest_rx = self.split_contest_exc(
                rst_received, db_values.get("mode")
            )
            db_values["contest_ex_rx"] = ctest_rx
            db_values["rst_received"] = rst_received
        rst_send = db_values.get("rst_send")
        if rst_send and not db_values.get("contest_ex_tx"):
            rst_send, ctest_tx = self.split_contest_exc(rst_send, db_values.get("mode"))
            db_values["contest_ex_tx"] = ctest_tx
            db_values["rst_send"] = rst_send
        db_values = self._add_times(db_values)
        call = db_values.get("dst_call")
        band = db_values.get("band")
        last_call = None
        last_band = None
        last_qso = self.db.last_qso()
        update = False
        if last_qso is not None:
            last_call = last_qso["dst_call"]
            last_band = last_qso["band"]
            update = bool(last_call == call) and bool(last_band == band)
        c_info = self.db.contest_info(curr_contest)
        if c_info is not None:
            print("Contest: {}".format(c_info.name))
            missing_attrs = []
            unique_vals = dict()
            for r_attr in c_info.required_attrs:
                val = db_values.get(r_attr)
                if not val:
                    missing_attrs.append(r_attr)
                if r_attr in c_info.unique_station_attrs:
                    unique_vals[r_attr] = val
            if missing_attrs:
                print(
                    "Contest QSO not complete. Missing {}".format(
                        ",".join(missing_attrs)
                    )
                )
                if call:
                    double_qso = self.db.contest_double_check(curr_contest, unique_vals)
                    if double_qso:
                        unique_l = [
                            "   {}: {}".format(k, v)
                            for k, v in unique_vals.items()
                            if k not in ["dst_call", "src_call", "contest_name"]
                        ]
                        unique = "\n".join(unique_l)
                        print("Double QSO with {} in contest\n{}".format(call, unique))
            else:
                band_id = "ALL"
                if c_info.number_per_band:
                    band_id = band
                # search for double qso
                double_qso = self.db.contest_double_check(curr_contest, unique_vals)
                if double_qso and not update:
                    print("Double contest qso")
                    self.show_qso(double_qso, False, show_locator=True)
                    num = self.db.get_contest_number(c_info.name, band_id) + 1
                else:
                    r = None
                    if update:
                        rowid = self.db.last_qso_number()
                        if rowid is not None:
                            self.db.update_qso(db_values, rowid)
                            num = self.db.get_contest_number(c_info.name, band_id)
                            r = self.db.last_qso()
                        else:
                            print("Row mismatch for update")
                    else:
                        self.db.insert_qso(db_values)
                        num = self.db.update_contest_number(c_info.name, band_id) + 1
                        r = self.db.current_qso()
                        self.db.commit_session()
                    if r:
                        self.show_qso(r, False, True)
                print("Next contest number: {:03d}".format(num))
        else:
            print("Current contest is unknown")

    def db_operations_normal(self, db_values, rig_info):
        """
        store data in db
        """
        db_values = self.upper_case(db_values)
        call = db_values.get("dst_call")
        is_uncommited = self.db.is_uncommited()
        do_insert = False
        if not is_uncommited and not call:
            return "Needs callsign first to log qso"
        if is_uncommited and call and call != self.db.current_call():
            do_insert = True
        if not is_uncommited:
            do_insert = True
        if do_insert:
            if self.check_complete():
                # Time and date only for new qso or if specified by command line
                db_values = self._add_defaults(db_values, rig_info)
                db_values = self._add_times(db_values)
                self.db.insert_qso(db_values)
                r = self.db.current_qso()
                if r:
                    self.show_qso(r, False)
                self.show_last_qsos(db_values["dst_call"])
        else:
            f_str = db_values.get("freq")
            f = None
            if f_str is not None:
                if type(f_str) == float:
                    f = f_str
                else:
                    f = self.frequ_in_hz(f_str)
            if f:
                db_values["freq"] = f
                band = frequency_to_band(f)
                db_values["band"] = band if band else ""
            self.db.update_qso(db_values)

            r = self.db.current_qso()
            if r:
                self.show_qso(r, False)
        return ""

    def show_distance(self, own_loc, loc):
        a = toLoc(own_loc)
        b = toLoc(loc)
        bearing = int(bearing_degrees(a, b))
        dist = int(distance_haversine(a, b) / 1e3)
        print("Bearing: %d Distance : %d" % (bearing, dist))

    def cmdline(self):
        self.args = self._parse_cmd_line()
        if bool(self.args.rig) and not self.args.update_record:
            rig_info = read_rig(self.args.rig)
            if not rig_info.isValid():
                rignames = "\n".join(rig_names())
                print(
                    "Cmd line rig '{}' unknown. Known rigs:\n{}".format(
                        self.args.rig, rignames
                    )
                )

        default_rig = self.config_values.get("defaultrig")
        if default_rig:
            rig_info = read_rig(default_rig)
            if not rig_info.isValid():
                print(
                    "Defaultrig entry '{}' in rigdesc.json\n does not exist or is invalid".format(
                        default_rig
                    )
                )
        else:
            rig_info = None
        if not self.handle_cmds():
            matched, unmatched = guess_args(self.args.cmdargs)
            for k, v in self.args.__dict__.items():
                if v is not None and k not in self._cmds:
                    if k == "rig":
                        # some special handling for rigs
                        # this inserts antenna and rig
                        if v:
                            rig_info = read_rig(v)
                            v = rig_info.get_rigdesc()
                    elif k == "cc":
                        if v:
                            # and for cc get call from clutserinfo
                            v = self.cluster_call()
                            if v:
                                k = "call"
                            else:
                                print("No call from cluster", file=sys.stderr)
                                continue
                        else:
                            continue
                    matched[k] = v
            if unmatched:
                # alles unbekannt geht in info
                matched["info"] = " ".join(unmatched)
            loc = matched.get("locator")
            own_loc = self.config_values.get("locator")
            if loc and own_loc:
                self.show_distance(own_loc, loc)
            db_values = self.ns_to_db_names(matched)
            if self.args.update_record is None:
                # add default values f not given
                self.db_operations(db_values, rig_info, self.args.contest)
            else:
                del db_values["update_record"]
                db_values = self.upper_case(db_values)
                self.db.update_qso(db_values, self.args.update_record)
        return 0

    def look_up_repeater(self, locator, freq=None, max_dist=None):
        """
        lookup repeater
        :param locator: str with locator
        :param freq: frequncy in Hz
        :param max_dist: maximum distance
        :return list of tuples (callsign, description)
        """
        repeater = []
        fnames = get_repeater_files()
        if fnames:
            rep = Repeater(fnames)
            ownPos = toLoc(locator)
            if freq is not None:
                f = freq / 1e6
            else:
                f = None
            distances = rep.by_distance(ownPos, f)
            d_sort = sorted(distances.keys())
            for i in range(len(d_sort)):
                d = d_sort[i]
                if max_dist is not None and d / 1e3 > max_dist:
                    break
                repeaters = distances[d]
                calls = ", ".join(
                    [
                        r["Frequency"]
                        + " "
                        + r["Mode"]
                        + ": "
                        + r["Name"]
                        + "/"
                        + r["Sub Name"]
                        for r in repeaters
                    ]
                )
                rep_desc = "Dist: {} km: {}".format(int(int(d) / 1e3), calls)
                callsign = None
                if repeaters:
                    callsign = repeaters[0]["Sub Name"]
                repeater.append((callsign, rep_desc))
        return repeater

    def print_repeater(self, arg):
        """
        print repeater info for given arg
        if arg is frequency in <number>[M|k|G]Hz
        print neares repeater for  given frequency
        If a locator is given print all repetares
        order by distance

        """
        named_args, _unnamed = guess_args([arg])
        data = read_config()
        loc = data.get("locator")
        f_hz = None
        f_str = named_args.get("freq")
        if f_str:
            f_hz = self.frequ_in_hz(f_str)
        else:
            loc = named_args.get("locator")
        if loc:
            rep_list = self.look_up_repeater(loc, f_hz, 120)
            if rep_list:
                for _call, info in rep_list:
                    print(info)
            else:
                print("No repeater found.")
        else:
            print("Need locator from config or as argument for repeater list")

    def add_repeater_from_list(self, db_values):
        # add repeater if possible
        rep = db_values.get("repeater")
        own_loc = db_values.get("src_locator")
        freq = db_values.get("freq")
        if rep is None and own_loc is not None and freq is not None:
            if freq > 29e6:
                repeaters = self.look_up_repeater(own_loc, freq)
                if repeaters:
                    db_values["repeater"] = repeaters[0][0]
        return db_values

    def qsl_record(self, request):
        """
        param: request str of pse or tnx
        mark given qso with qslin and sets qslreq if necessary
        if request == pse and qslout is not set, set qslrequest
        """
        # look for qsos with params
        matched, _unmatched = guess_args(self.args.cmdargs)
        if self.args.call:
            matched["call"] = self.args.call
        db_values = self.ns_to_db_names(matched)
        # search qsos for call and date
        row_ids = {}
        for row in self.db.search_qso_by_attrs(db_values):
            row_ids[row["rowid"]] = row
            self.show_qso(row, True, True)
        cnt_rows = len(row_ids)
        if cnt_rows > 0:
            qso_row = None
            if cnt_rows > 1:
                selected_qso = None
                while selected_qso is None:
                    try:
                        qso_num = input("Select QSO by row: ")
                    except KeyboardInterrupt:
                        return
                    if qso_num.isdigit() and int(qso_num) in row_ids:
                        selected_qso = int(qso_num)
                    else:
                        print("Invalid number. Use CTRL c to exit")
                qso_row = row_ids.get(selected_qso)
            else:
                qso_row = list(row_ids.values())[0]
            if qso_row is not None:
                update_attrs = {}
                if request.lower() == "pse" and not qso_row["qslout"]:
                    update_attrs["qslreq"] = "qslreq"
                update_attrs["qslin"] = "qslin"
                self.db.update_qso(update_attrs, qso_row["rowid"])
        else:
            print("No match")


def _main():
    try:
        logbook = BbpLog()
        ret = logbook.cmdline()
    except ConfigException:
        ret = 1
    return ret


if __name__ == "__main__":
    sys.exit(_main())
